package main

import (
	"net"
	"os"
	"os/signal"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/legend/cmd/legend/api"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/xray/plugins/beanstalk"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/vod/vinyl/client"
	"code.justin.tv/web/channel-audits/client"
)

const (
	teamName    = "feeds"
	serviceName = "legend"
)

type service struct {
	osExit        func(code int)
	onListen      func(net.Addr)
	Runner        service_common.ServiceRunner
	serviceCommon service_common.ServiceCommon
	sigChan       chan os.Signal

	apiService api.HTTPService

	vinylClient         vinyl.Client
	channelAuditsClient channel_audits.Client

	configs struct {
		HTTPConfig          service_common.BaseHTTPServerConfig
		VinylConfig         vinylConfig
		ChannelAuditsConfig channelAuditsConfig
	}
}

type channelAuditsConfig struct {
	host *distconf.Str
}

func (c *channelAuditsConfig) Load(d *distconf.Distconf) error {
	c.host = d.Str("legend.clients.channel_audits.host", "")
	return nil
}

type vinylConfig struct {
	host *distconf.Str
}

func (v *vinylConfig) Load(d *distconf.Distconf) error {
	v.host = d.Str("legend.clients.vinyl.host", "")
	return nil
}

func (s *service) setupConfigs() error {
	if err := service_common.LoadConfigs(
		s.serviceCommon.Config,
		&s.configs.VinylConfig,
		&s.configs.ChannelAuditsConfig,
	); err != nil {
		return err
	}

	if err := s.configs.HTTPConfig.Verify(s.serviceCommon.Config, "legend"); err != nil {
		return err
	}

	if err := s.serviceCommon.XRay.WithPlugin(&beanstalk.Plugin{}); err != nil {
		s.serviceCommon.Log.Log("err", err, "unable to enable beanstalk plugin")
	}

	return nil
}

func (s *service) setup() error {
	err := s.serviceCommon.Setup()
	if err != nil {
		return err
	}

	err = s.setupConfigs()
	if err != nil {
		return err
	}

	vinylClient, err := vinyl.NewClient(twitchclient.ClientConf{
		Host: s.configs.VinylConfig.host.Get(),
	})
	if err != nil {
		return err
	}
	s.vinylClient = vinylClient

	channelAuditsClient, err := channel_audits.NewClient(twitchclient.ClientConf{
		Host: s.configs.ChannelAuditsConfig.host.Get(),
	})
	if err != nil {
		return err
	}
	s.channelAuditsClient = channelAuditsClient

	return nil
}

func (s *service) inject() {

	s.apiService = api.HTTPService{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &s.configs.HTTPConfig,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Dims:        &s.serviceCommon.CtxDimensions,
			Log:         s.serviceCommon.Log,
			ElevateKey:  s.serviceCommon.ElevateLogKey,
			Ctxlog:      &s.serviceCommon.Ctxlog,
			OnListen:    s.onListen,
			PanicLogger: s.serviceCommon.PanicLogger,
			XRay:        s.serviceCommon.XRay,
		},

		Vinyl:         s.vinylClient,
		ChannelAudits: s.channelAuditsClient,
	}

	s.apiService.SetupRoutes = s.apiService.LegendRoutes

	s.Runner = service_common.ServiceRunner{
		Log: s.serviceCommon.Log,
		Services: []service_common.Service{
			&s.apiService,
		},
		SigChan:      s.sigChan,
		SignalNotify: signal.Notify,
	}
}

func (s *service) main() {
	if err := s.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		s.osExit(1)
		return
	}

	s.inject()

	if err := s.Runner.Execute(); err != nil {
		service_common.SetupLogger.Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.serviceCommon.Log.Log("done with service")
}

func main() {
	baseDirectory := ""
	if os.Getenv("ENVIRONMENT") != "development" {
		baseDirectory = "/etc/legend/"
	}
	instance := service{
		osExit: os.Exit,
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
				BaseDirectory: baseDirectory,
			},
		},
	}

	instance.main()
}
