package xray

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net"
)

var header = []byte(`{"format": "json", "version": 1}` + "\n")

func (x *XRay) refreshEmitter(udpaddr *net.UDPAddr) error {
	var err error
	x.emitter, err = net.DialUDP("udp", nil, udpaddr)
	if err != nil {
		x.emitter = ioutil.Discard
		return err
	}
	return nil
}

func (x *XRay) encodePieces(pieces [][]byte) {
	for _, p := range pieces {
		_, err := x.emitter.Write(append(header, p...))
		if err != nil {
			x.onError("unable to emit segment to UDP socket", err)
		}
	}
}

func (x *XRay) emit(seg *segment) {
	if seg == nil || !seg.sampled {
		return
	}

	allSpareBuffers := make([]*bytes.Buffer, 0, 4)
	// Using a byte buffer pool means one more alloc/op (unless we are writing multiple segments), but
	// about 2/3 the byte usage.  See BenchmarkXRay_Capture
	newByteBuffer := func() *bytes.Buffer {
		b := byteBufferPool.Get().(*bytes.Buffer)
		allSpareBuffers = append(allSpareBuffers, b)
		return b
	}

	pieces := x.encodeIntoPieces(seg, nil, newByteBuffer)
	x.encodePieces(pieces)

	for _, b := range allSpareBuffers {
		// We can only add these back at the end because that's when I can Reset() and reuse their bytes
		b.Reset()
		byteBufferPool.Put(b)
	}
}

func (x *XRay) encodeIntoPieces(seg *segment, out [][]byte, newByteBuffer func() *bytes.Buffer) [][]byte {
	seg.Lock()
	defer seg.Unlock()

	encode := func(s *segment) []byte {
		buf := newByteBuffer()
		if err := json.NewEncoder(buf).Encode(s); err != nil {
			x.onError("unable to encode segment", err)
			return nil
		}
		b := buf.Bytes()

		// While the segment is too big, remove children to make it smaller
	TrimLoop:
		for len(b) > maxSegmentSize {
			// There are no children left to remove
			if len(s.rawSubsegments) == 0 {
				// Try to remove stack trace instead
				if s.Cause != nil {
					s.Cause.Exceptions = nil
					x.msg(fmt.Sprintf("Stripping stack-trace for %s:%s to reduce segment size", s.root().TraceID, s.ID))
					continue TrimLoop
				}

				// We've run out of options
				x.msg(fmt.Sprintf("Failed to emit %s:%s, size (%dB) was greater than the %dB limit", s.root().TraceID, s.ID, len(b), maxSegmentSize))
				return nil
			}

			// Find the largest subsegment
			// By splitting the largest subsegment we minimize the number of splits
			size, idx := 0, 0
			for i, b := range s.Subsegments {
				if len(b) > size {
					size, idx = len(b), i
				}
			}

			// Remove the subsegment from the subsegments list
			child := s.rawSubsegments[idx]
			s.Subsegments = append(s.Subsegments[:idx], s.Subsegments[idx+1:]...)
			s.rawSubsegments = append(s.rawSubsegments[:idx], s.rawSubsegments[idx+1:]...)

			// Add extra information into child subsegment, re-marshal, and add to out
			child.Lock()
			child.TraceID = s.root().TraceID
			child.ParentID = s.ID
			child.Type = "subsegment"
			child.parent = nil
			child.sampled = s.sampled
			child.requestWasTraced = s.requestWasTraced

			cb, _ := json.Marshal(child)
			out = append(out, cb)
			child.Unlock()

			// Recompute current segment's encoded form
			var err error
			b, err = json.Marshal(s)
			if err != nil {
				x.onError("unable to marshal segment", err)
				return nil
			}
		}

		return b
	}

	for _, s := range seg.rawSubsegments {
		out = x.encodeIntoPieces(s, out, newByteBuffer)
		if b := encode(s); b != nil {
			seg.Subsegments = append(seg.Subsegments, b)
		}
	}
	if seg.parent == nil {
		if b := encode(seg); b != nil {
			out = append(out, b)
		}
	}
	return out
}
