package manager

import (
	"crypto/rand"
	"encoding/base64"
	"errors"
	"fmt"
	"path"
	"strconv"
	"strings"
	"time"

	ssErrors "code.justin.tv/systems/sandstorm/errors"
	"github.com/asaskevich/govalidator"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

// SecretClass Type for secret Category
type SecretClass int8

const (
	// ClassUnknown Secret which have not been classfified yet.
	ClassUnknown SecretClass = iota //0
	// ClassPublic :  1
	ClassPublic
	// ClassPrivate : 2
	ClassPrivate
	// ClassConfidential : 3
	ClassConfidential
	// ClassSecret : 4
	ClassSecret
	// ClassTopSecret : 5
	ClassTopSecret
)

const (
	// PlaintextLengthMin is the minimum length that a plaintext secret can be
	// when generated using FillPlaintext
	PlaintextLengthMin = 2
	// PlaintextLengthMax is the maxmimum length that a plaintext secret can be
	// when generated using FillPlaintext
	PlaintextLengthMax = 4096
)

const jsonClassField = "class"

// Secret represents a secret. It can be either in the plain or
// enciphered.
type Secret struct {
	// Name of secret
	Name string `json:"name"`
	// Plaintext to be enciphered or read by user
	Plaintext []byte `json:"plaintext,omitempty"`
	// Unix timestamp of when the key was created/updated
	UpdatedAt int64 `json:"updated_at"`
	// Ciphertext to be deciphered or uploaded
	DoNotBroadcast bool `json:"do_not_broadcast"`
	// Tombstone signifies that a secret has been deleted
	Tombstone bool `json:"tombstone"`
	// CrossEnv signifies that a secret can be used in any environment
	CrossEnv bool `json:"cross_env"`
	// KeyARN is the key used to encrypt
	KeyARN string `json:"key_arn"`

	// Class of the document. update const jsonClassField if you change the json field name.
	Class SecretClass `json:"class"`

	ciphertext []byte
	// Enciphered key returned by kms.GenerateDataKey
	key []byte
}

func (s *Secret) asDynamoSecret() *dynamoSecret {
	return &dynamoSecret{
		Tombstone:      s.Tombstone,
		Name:           s.Name,
		UpdatedAt:      s.UpdatedAt,
		Key:            base64.StdEncoding.EncodeToString(s.key),
		Value:          base64.StdEncoding.EncodeToString(s.ciphertext),
		KeyARN:         s.KeyARN,
		DoNotBroadcast: s.DoNotBroadcast,
		Class:          int(s.Class),
	}
}

type dynamoSecret struct {
	Tombstone      bool   `dynamodbav:"tombstone"`
	Name           string `dynamodbav:"name" valid:"required"`
	UpdatedAt      int64  `dynamodbav:"updated_at" valid:"required"`
	Value          string `dynamodbav:"value" valid:"required"`
	Key            string `dynamodbav:"key" valid:"required"`
	KeyARN         string `dynamodbav:"key_arn"`
	DoNotBroadcast bool   `dynamodbav:"do_not_broadcast"`
	Class          int    `dynamodbav:"class"`
	CrossEnv       bool   `dynamodbav:"cross_env"`
}

func unmarshalSecret(item map[string]*dynamodb.AttributeValue) (secret *Secret, err error) {
	var s dynamoSecret
	err = dynamodbattribute.UnmarshalMap(item, &s)
	if err != nil {
		return
	}

	if s.Tombstone {
		err = ssErrors.ErrSecretTombstoned
		return
	}

	_, err = govalidator.ValidateStruct(s)
	if err != nil {
		err = fmt.Errorf("error validating secret from dynamodb: %s", err.Error())
		return
	}

	secret = &Secret{
		Name:           s.Name,
		UpdatedAt:      s.UpdatedAt,
		DoNotBroadcast: s.DoNotBroadcast,
		KeyARN:         s.KeyARN,
		Tombstone:      s.Tombstone,
		Class:          SecretClass(s.Class),
		CrossEnv:       s.CrossEnv,
	}

	ciphertext, err := base64.StdEncoding.DecodeString(s.Value)
	if err != nil {
		err = fmt.Errorf("error base64 decoding ciphertext for secret %s: %s", secret.Name, err.Error())
		return
	}
	secret.ciphertext = ciphertext

	key, err := base64.StdEncoding.DecodeString(s.Key)
	if err != nil {
		err = fmt.Errorf("error base64 decoding key for secret %s: %s", secret.Name, err.Error())
		return
	}
	secret.key = key

	return
}

// FillPlaintextRequest used for FillPlaintext
type FillPlaintextRequest struct {
	Length    int `json:"length"`
	validated bool
}

// Validate Length
func (r FillPlaintextRequest) Validate() error {
	if r.validated {
		return nil
	}
	if r.Length < PlaintextLengthMin {
		return errors.New("plaintext length is less than minumum length")
	}
	if r.Length > PlaintextLengthMax {
		return errors.New("plaintext length is greater than maximum length")
	}
	r.validated = true
	return nil
}

// FillPlaintext fills the Plaintext field with a random byte combination
// that is base64 decodable (i.e. guaranteed printable). Note that this wil be
// double encoded in Sandstorm -- this is intentional!
func (s *Secret) FillPlaintext(r *FillPlaintextRequest) error {
	if err := r.Validate(); err != nil {
		return err
	}
	plaintextRaw := make([]byte, r.Length)
	_, err := rand.Read(plaintextRaw)
	if err != nil {
		return err
	}

	// base64 decodable since we want to be printable.
	s.Plaintext = make([]byte, base64.StdEncoding.EncodedLen(r.Length))
	base64.StdEncoding.Encode(s.Plaintext, plaintextRaw)

	return nil
}

// Timestamp updates the UpdatedAt field to current unix timestamp
func (s *Secret) Timestamp() {
	s.UpdatedAt = time.Now().Unix()
}

// Seal takes a secret, timestamps it, creates an encryption context
// and passes this with m.Config.KeyID to envelope.Seal
func (m *Manager) Seal(secret *Secret) error {
	secret.Timestamp()
	key, ciphertext, keyARN, err := m.Envelope.Seal(m.Config.KeyID, secret.EncryptionContext(), secret.Plaintext)
	if err != nil {
		return err
	}
	// XXX perform base64 here?
	secret.key = key
	secret.ciphertext = ciphertext
	Zero(secret.Plaintext)
	secret.Plaintext = nil
	secret.KeyARN = keyARN
	return nil
}

// Decrypt the Ciphertext field and store it in Plaintext
func (m *Manager) Decrypt(secret *Secret) error {
	if secret == nil {
		return nil
	}

	plaintext, err := m.Envelope.Open(secret.key, secret.ciphertext, secret.EncryptionContext())
	if err != nil {
		return err
	}
	secret.Plaintext = plaintext
	return nil
}

// EncryptionContext generates the encryption context map to pass to
// kms.GenerateDataKey / kms.Decrypt.
func (s *Secret) EncryptionContext() (context map[string]string) {
	context = map[string]string{
		"name":       s.Name,
		"updated_at": strconv.FormatInt(s.UpdatedAt, 10),
	}
	return
}

// SecretNamespace contains information about the different parts of the name of a secret.
type SecretNamespace struct {
	Team        string `json:"team"`
	System      string `json:"system"`
	Environment string `json:"environment"`
	Name        string `json:"name"`
}

func (s SecretNamespace) String() string { return path.Join(s.Team, s.System, s.Environment, s.Name) }

// ParseSecretNamespace extracts namespace information from the name of a secret.
func ParseSecretNamespace(secretName string) (*SecretNamespace, error) {
	splitted := strings.SplitN(secretName, "/", 4)
	if len(splitted) != 4 {
		return nil, ssErrors.InvalidRequestError{Message: fmt.Errorf("sandstorm: expected 4 namespace elements, got %d", len(splitted))}
	}
	sn := &SecretNamespace{
		Team:        splitted[0],
		System:      splitted[1],
		Environment: splitted[2],
		Name:        splitted[3],
	}
	return sn, nil
}
