package vinyl

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "vinyl"
)

// Client is the interface a Vinyl client implements
type Client interface {
	GetVodByID(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) (*Vod, error)
	GetVodByIDVisage(ctx context.Context, id string, params *GetVodParams, reqOpts *twitchclient.ReqOpts) (*Vod, error)
	GetVodsByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchclient.ReqOpts) ([]*Vod, error)
	GetVodsAggregationByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchclient.ReqOpts) (*VodsAggregation, error)
	GetVodsByQuery(ctx context.Context, params GetVodsByQueryParams, reqOpts *twitchclient.ReqOpts) ([]*Vod, error)
	UpdateVod(ctx context.Context, id string, params *UpdateVodInput, reqOpts *twitchclient.ReqOpts) (*Vod, error)
	Moderation(ctx context.Context, vodID, status string, reqOpts *twitchclient.ReqOpts) error
	DeleteVodExternal(ctx context.Context, vodID string, reqOpts *twitchclient.ReqOpts) error
	DeleteVodsExternal(ctx context.Context, vodIDs []string, reqOpts *twitchclient.ReqOpts) error
	Top(ctx context.Context, input TopInput, reqOpts *twitchclient.ReqOpts) ([]*Vod, error)
	Followed(ctx context.Context, ids []string, input FollowedInput, reqOpts *twitchclient.ReqOpts) ([]*Vod, error)
	GetVodsByUser(ctx context.Context, userID string, input GetVodsByUserInput, reqOpts *twitchclient.ReqOpts) (*GetVodsByUserResponse, error)
	GetUserVodProperties(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*UserVODProperties, error)
	CreateHighlight(ctx context.Context, input CreateHighlightInput, reqOpts *twitchclient.ReqOpts) (*Vod, error)
	Rebroadcast(ctx context.Context, vodIDs []string, ownerID, streamKey string, reqOpts *twitchclient.ReqOpts) error
	CreateAppeals(ctx context.Context, input CreateAppealsInput, reqOpts *twitchclient.ReqOpts) error
	YoutubeExport(ctx context.Context, input YoutubeExportInput, reqOpts *twitchclient.ReqOpts) error
	SearchVods(ctx context.Context, userID, search, limit string, reqOpts *twitchclient.ReqOpts) ([]*Vod, error)
	LiveToVod(ctx context.Context, streamID, userID string, reqOpts *twitchclient.ReqOpts) (*LiveToVodResponse, error)
}

type client struct {
	twitchclient.Client
}

// NewClient takes a twitchclient.ClientConf and returns a client implementing the Vinyl client interface
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}

type getVodsByIDsResponse struct {
	TotalCount int    `json:"total_count"`
	Vods       []*Vod `json:"vods"`
}

// GetVodsByUserResponse is the response of the /v1/vods/user/:userID endpoint
type GetVodsByUserResponse struct {
	TotalCount int    `json:"total_count"`
	Vods       []*Vod `json:"vods"`
}

// VodList is a list of vods
type VodList []*Vod

type createHighlightResponse struct {
	Vod *Vod `json:"highlight"`
}

// GetVodParams will translate into request parameters used when loading /v1/vods
// Please use NewGetVodParams to instantiate with default values.
type GetVodParams struct {
	WithMuteInfo             bool // (appeals_and_amrs) response with mute info fields (i.e. muted_segments)
	WithNotificationSettings bool // (notification_settings) response with notification_settings fields
	IncludeDeleted           bool // (include_deleted)
	IncludePrivate           bool // (include_private)
	IncludeProcessing        bool // (include_processing)
}

// NewGetVodParams instantiates GetVodParams with defaults
func NewGetVodParams() *GetVodParams {
	return &GetVodParams{
		WithMuteInfo:             false, // since needs extra effort to load this data
		WithNotificationSettings: false,
		IncludeDeleted:           false, // soft-removed vods are ignored by default
		IncludePrivate:           true,
		IncludeProcessing:        true,
	}
}

// URLQueryString converts a GetVodParam struct into the expected query parameters for the vinyl backend
func (p *GetVodParams) URLQueryString() string {
	qs := fmt.Sprintf(
		"appeals_and_amrs=%t&notification_settings=%t&include_deleted=%t&include_private=%t&include_processing=%t",
		p.WithMuteInfo,
		p.WithNotificationSettings,
		p.IncludeDeleted,
		p.IncludePrivate,
		p.IncludeProcessing,
	)
	return qs
}

func (c *client) GetVodByID(ctx context.Context, idStr string, reqOpts *twitchclient.ReqOpts) (*Vod, error) {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/%s", idStr), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vod_by_id",
		StatSampleRate: defaultStatSampleRate,
	})

	vod := Vod{}
	resp, err := c.DoJSON(ctx, &vod, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode == 404 {
		return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "vod not found"}
	}

	return &vod, nil
}

func (c *client) GetVodByIDVisage(ctx context.Context, idStr string, params *GetVodParams, reqOpts *twitchclient.ReqOpts) (*Vod, error) {
	if params == nil {
		params = &GetVodParams{} // defaults
	}

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/visage/vods/%s?%s", idStr, params.URLQueryString()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vod_by_id_visage",
		StatSampleRate: defaultStatSampleRate,
	})

	vod := Vod{}
	resp, err := c.DoJSON(ctx, &vod, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode == 404 {
		return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "vod not found"}
	}

	return &vod, nil
}

// GetVodsByQueryParams will translate into request parameters used when loading /v1/vods_by_query
type GetVodsByQueryParams struct {
	BroadcastIDs *string
}

// URLQueryString converts a GetVodsByQueryParams struct into the expected query parameters for the vinyl backend
func (p GetVodsByQueryParams) URLQueryString() string {
	v := url.Values{}
	if p.BroadcastIDs != nil {
		v.Add("broadcast_ids", *p.BroadcastIDs)
	}

	return v.Encode()
}

type getVodsByQueryResponse struct {
	Vods []*Vod `json:"vods"`
}

func (c *client) GetVodsByQuery(ctx context.Context, params GetVodsByQueryParams, reqOpts *twitchclient.ReqOpts) ([]*Vod, error) {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods_by_query?%s", params.URLQueryString()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vods_by_query",
		StatSampleRate: defaultStatSampleRate,
	})

	data := getVodsByQueryResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return data.Vods, nil
}

func (c *client) GetVodsByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchclient.ReqOpts) ([]*Vod, error) {
	if params == nil {
		params = &GetVodParams{} // defaults
	}
	idsStr := strings.Join(ids, ",")

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods?ids=%s&%s", idsStr, params.URLQueryString()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vods_by_ids",
		StatSampleRate: defaultStatSampleRate,
	})

	data := getVodsByIDsResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if len(data.Vods) == 0 {
		return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "vods not found"}
	}

	return data.Vods, nil
}

func (c *client) GetVodsAggregationByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchclient.ReqOpts) (*VodsAggregation, error) {
	if params == nil {
		params = &GetVodParams{} // defaults
	}
	idsStr := strings.Join(ids, ",")

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods_aggregation?ids=%s&%s", idsStr, params.URLQueryString()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vods_aggregation_by_ids",
		StatSampleRate: defaultStatSampleRate,
	})

	data := VodsAggregation{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &data, nil
}

// UpdateVodInput contains the possible input parameters that go into a UpdateVod operation.
type UpdateVodInput struct {
	Description           *string                       `json:"description"`
	Game                  *string                       `json:"game"`
	Language              *string                       `json:"language"`
	Notifications         UpdateVodNotificationSettings `json:"notifications"`
	SelectedThumbnailPath *string                       `json:"selected_thumbnail_path"`
	TagList               *string                       `json:"tag_list"`
	Title                 *string                       `json:"title"`
	// Can be "private" or "public".
	// "private" means this vod is only viewable by owners.
	// "public" means it is viewable to the public.
	Viewable   *string    `json:"viewable"`
	ViewableAt *time.Time `json:"viewable_at"`
}

// VodNotificationInput contains the possible input parameters for notification settings
type VodNotificationInput struct {
	CustomText *string    `json:"custom_text"`
	Enabled    bool       `json:"enabled"`
	SentAt     *time.Time `json:"sent_at"`
}

// UpdateVodNotificationSettings contains allowed VodNotifications to update
type UpdateVodNotificationSettings struct {
	Email       *VodNotificationInput `json:"email"`
	ChannelFeed *VodNotificationInput `json:"channel_feed"`
}

type updateVodResponse struct {
	Vod *Vod `json:"vod"`
}

func (c *client) UpdateVod(ctx context.Context, id string, params *UpdateVodInput, reqOpts *twitchclient.ReqOpts) (*Vod, error) {
	marshalledParams, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPut, fmt.Sprintf("/v1/vods/%v", id), bytes.NewBuffer(marshalledParams))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.update_vod",
		StatSampleRate: defaultStatSampleRate,
	})

	data := updateVodResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return data.Vod, nil
}

func (c *client) DeleteVodsExternal(ctx context.Context, vodIDs []string, reqOpts *twitchclient.ReqOpts) error {
	vodIDsStr := strings.Join(vodIDs, ",")
	req, err := c.NewRequest(http.MethodDelete, fmt.Sprintf("/v1/vods/external?ids=%s", vodIDsStr), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.delete_vods_external",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
		}
	}()

	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *client) DeleteVodExternal(ctx context.Context, vodID string, reqOpts *twitchclient.ReqOpts) error {
	req, err := c.NewRequest(http.MethodDelete, fmt.Sprintf("/v1/vods/external?ids=%s", vodID), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.delete_vod_external",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
		}
	}()

	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

func (c *client) Moderation(ctx context.Context, vodID, status string, reqOpts *twitchclient.ReqOpts) error {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/moderation/%s?status=%s", vodID, status), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.moderate_vod",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
		}
	}()

	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}

// TopInput contains the possible input parameters that go into a Top operation.
type TopInput struct {
	// comma-separated list containing which vod broadcast types to filter for
	//   "archive", "highlight", "upload" are the available broadcast_types
	//   Example values: "upload", "archive,highlight", "archive,highlight,upload"
	//   Passing in an empty string defaults to no filter.
	BroadcastType string
	// comma-separated list containing languages to filter for
	//   e.g. "es,en"
	//   Passing in an empty string defaults to no filter.
	Language string
	// string containing game to filter for
	//   e.g. "Hearthstone"
	//   Passing in an empty string defaults to no filter.
	Game string
	// Filter for vods within a specified interval. Can be one of "all", "month", "week", "day"
	Period string
	// Which field to sort desc on. Can be only "views" or "time"
	Sort   string
	Limit  int
	Offset int
}

func (c *client) Top(ctx context.Context, input TopInput, reqOpts *twitchclient.ReqOpts) ([]*Vod, error) {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/top?broadcast_type=%s&language=%s&game=%s&period=%s&sort=%s&limit=%d&offset=%d", input.BroadcastType, input.Language, url.QueryEscape(input.Game), input.Period, input.Sort, input.Limit, input.Offset), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.top",
		StatSampleRate: defaultStatSampleRate,
	})

	data := VodList{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

// FollowedInput contains the possible input parameters that go into a Followed operation.
type FollowedInput struct {
	// comma-separated list containing which vod broadcast types to filter for
	//   "archive", "highlight", "upload" are the available broadcast_types
	//   Example values: "upload", "archive,highlight", "archive,highlight,upload"
	//   Passing in an empty string defaults to no filter.
	BroadcastType string
	// comma-separated list containing languages to filter for
	//   e.g. "es,en"
	//   Passing in an empty string defaults to no filter.
	Language string
	// Which field to sort desc on. Can be only "views" or "time"
	Sort   string
	Limit  int
	Offset int
}

func (c *client) Followed(ctx context.Context, ids []string, input FollowedInput, reqOpts *twitchclient.ReqOpts) ([]*Vod, error) {
	var idsAsInt []int

	for _, stringID := range ids {
		id, err := strconv.Atoi(stringID)
		if err == nil {
			idsAsInt = append(idsAsInt, id)
		}
	}

	i, err := json.Marshal(idsAsInt)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, fmt.Sprintf("/v1/vods/followed?broadcast_type=%s&language=%s&sort=%s&limit=%d&offset=%d", input.BroadcastType, input.Language, input.Sort, input.Limit, input.Offset), bytes.NewBuffer(i))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.followed",
		StatSampleRate: defaultStatSampleRate,
	})

	data := VodList{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) CreateHighlight(ctx context.Context, input CreateHighlightInput, reqOpts *twitchclient.ReqOpts) (*Vod, error) {
	i, err := json.Marshal(map[string]CreateHighlightInput{"highlight": input})
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, "/v1/vods/highlight", bytes.NewBuffer(i))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.create_highlight",
		StatSampleRate: defaultStatSampleRate,
	})

	data := createHighlightResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data.Vod, err
}

// CreateHighlightInput contains the possible input parameters that go into a CreateHighlight operation.
type CreateHighlightInput struct {
	VodID        int64  `json:"id"`
	Description  string `json:"description"`
	Game         string `json:"game"`
	Title        string `json:"title"`
	Language     string `json:"language"`
	StartSeconds int    `json:"start_time"`
	EndSeconds   int    `json:"end_time"`
	TagList      string `json:"tag_list"`
	CreatedBy    int64  `json:"created_by_id"`
}

func (c *client) CreateAppeals(ctx context.Context, input CreateAppealsInput, reqOpts *twitchclient.ReqOpts) error {
	i, err := json.Marshal(input)
	if err != nil {
		return err
	}

	req, err := c.NewRequest(http.MethodPost, "/v1/appeals", bytes.NewBuffer(i))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.create_appeals",
		StatSampleRate: defaultStatSampleRate,
	})

	data := struct{}{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return err
}

// CreateAppealsInput is the input format associated with the CreateAppeals endpoint.
type CreateAppealsInput struct {
	VodID        string        `json:"vod_id"`
	VodAppeal    VodAppeal     `json:"vod_appeal_params"`
	TrackAppeals []TrackAppeal `json:"track_appeals"`
}

func (c *client) YoutubeExport(ctx context.Context, input YoutubeExportInput, reqOpts *twitchclient.ReqOpts) error {
	i, err := json.Marshal(input)
	if err != nil {
		return err
	}

	req, err := c.NewRequest(http.MethodPost, "/v1/vods/:vod_id/youtube_export", bytes.NewBuffer(i))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.youtube_export",
		StatSampleRate: defaultStatSampleRate,
	})

	data := struct{}{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return err
}

// YoutubeExportInput is the input format associated with the YoutubeExport endpoint.
type YoutubeExportInput struct {
	VodID       string `json:"vod_id"`
	Title       string `json:"title"`
	Description string `json:"description"`
	TagList     string `json:"tag_list"`
	Private     bool   `json:"private"`
	DoSplit     bool   `json:"do_split"`
}

// GetVodsByUserInput contains the possible input parameters that go into a GetVodsByUser operation.
type GetVodsByUserInput struct {
	// comma-separated list containing which vod broadcast types to filter for
	//   "archive", "highlight", "upload" are the available broadcast_types
	//   Example values: "upload", "archive,highlight", "archive,highlight,upload"
	//   Passing in an empty string defaults to no filter.
	BroadcastType string
	// comma-separated list containing languages to filter for
	//   e.g. "es,en"
	//   Passing in an empty string defaults to no filter.
	Language string
	// comma-separated list containing statuses to filter for
	//   e.g. "recording,recorded"
	//   Passing in an empty string defaults to no filter.
	Status string
	// Which field to sort desc on. Can be only "views" or "time"
	Sort string
	// filter for vods that were published within a specified interval.
	//   It accepts values in the format NUMBER UNIT, where number is a non-negative integer
	//   and UNIT is day/month/year. Combinations of different units is allowed, but each unit can only
	//   be specified at most once. Passing in an empty string filters for all vods.
	PublishedWithin string
	Limit           int
	Offset          int
	// true if this call should obey the user's hide_archives user_vod_property
	RespectHideArchives bool
	// privileged options
	GetVodParams *GetVodParams
}

func (c *client) GetVodsByUser(ctx context.Context, userID string, input GetVodsByUserInput, reqOpts *twitchclient.ReqOpts) (*GetVodsByUserResponse, error) {
	if input.GetVodParams == nil {
		input.GetVodParams = NewGetVodParams() // defaults
	}

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/user/%s", userID), nil)
	if err != nil {
		return nil, err
	}

	q := req.URL.Query()
	q.Add("broadcast_type", input.BroadcastType)
	q.Add("language", input.Language)
	q.Add("status", input.Status)
	q.Add("sort", input.Sort)
	q.Add("published_within", input.PublishedWithin)
	q.Add("limit", strconv.Itoa(input.Limit))
	q.Add("offset", strconv.Itoa(input.Offset))
	q.Add("respect_hide_archives", fmt.Sprintf("%t", input.RespectHideArchives))
	q.Add("appeals_and_amrs", fmt.Sprintf("%t", input.GetVodParams.WithMuteInfo))
	q.Add("notification_settings", fmt.Sprintf("%t", input.GetVodParams.WithNotificationSettings))
	q.Add("include_deleted", fmt.Sprintf("%t", input.GetVodParams.IncludeDeleted))
	q.Add("include_private", fmt.Sprintf("%t", input.GetVodParams.IncludePrivate))
	q.Add("include_processing", fmt.Sprintf("%t", input.GetVodParams.IncludeProcessing))
	req.URL.RawQuery = q.Encode()

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vods_by_user",
		StatSampleRate: defaultStatSampleRate,
	})

	data := GetVodsByUserResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &data, nil
}

type getUserVodPropertiesResponse struct {
	Properties UserVODProperties `json:"properties"`
}

func (c *client) GetUserVodProperties(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*UserVODProperties, error) {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/user_vod_properties/%s", userID), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_user_vod_properties",
		StatSampleRate: defaultStatSampleRate,
	})

	data := getUserVodPropertiesResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return &data.Properties, err
}

func (c *client) Rebroadcast(ctx context.Context, ids []string, ownerID, streamKey string, reqOpts *twitchclient.ReqOpts) error {
	idsStr := strings.Join(ids, ",")
	req, err := c.NewRequest(http.MethodPost, fmt.Sprintf("/v1/vods/rebroadcast?ids=%s&owner_id=%s&stream_key=%s", idsStr, ownerID, streamKey), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.rebroadcast",
		StatSampleRate: defaultStatSampleRate,
	})

	data := struct{}{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return err
}

func (c *client) SearchVods(ctx context.Context, userID, search, limit string, reqOpts *twitchclient.ReqOpts) ([]*Vod, error) {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/users/%s/vods?search=%s&limit=%s", userID, search, limit), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.search_vods",
		StatSampleRate: defaultStatSampleRate,
	})

	data := VodList{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

// LiveToVodResponse is the response format of the live-to-vod endpoint
type LiveToVodResponse struct {
	Stream  *UsherLivenessResponse `json:"stream"`
	Channel Channel                `json:"channel"`
	Vods    []*Vod                 `json:"vods"`
}

// UsherLivenessResponse is the response format from the usher livecheck endpoint
type UsherLivenessResponse struct {
	Channel   string `json:"channel"`
	ChannelID int64  `json:"channel_id"`
	SessionID string `json:"session_id"`
	StartedOn int64  `json:"started_on"`
	StreamID  int64  `json:"stream_id"`
	UpdatedOn int64  `json:"updated_on"`
}

// Channel contains the metadata for a channel. It is modeled after the channel model in Visage.
type Channel struct {
	Mature                       bool      `json:"mature"`
	Status                       string    `json:"status"`
	BroadcasterLanguage          string    `json:"broadcaster_language"`
	DisplayName                  string    `json:"display_name"`
	Game                         string    `json:"game"`
	Language                     string    `json:"language"`
	ID                           int       `json:"_id"`
	Name                         string    `json:"name"`
	CreatedAt                    time.Time `json:"created_at"`
	UpdatedAt                    time.Time `json:"updated_at"`
	Partner                      bool      `json:"partner"`
	Logo                         *string   `json:"logo"`
	VideoBanner                  *string   `json:"video_banner"`
	ProfileBanner                *string   `json:"profile_banner"`
	ProfileBannerBackgroundColor *string   `json:"profile_banner_background_color"`
	URL                          string    `json:"url"`
	Views                        int       `json:"views"`
	Followers                    int       `json:"followers"`
	BroadcasterType              string    `json:"broadcaster_type"`
	Description                  string    `json:"description"`
}

func (c *client) LiveToVod(ctx context.Context, streamID, userID string, reqOpts *twitchclient.ReqOpts) (*LiveToVodResponse, error) {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/streams/%s/channel/%s", streamID, userID), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.live_to_vod",
		StatSampleRate: defaultStatSampleRate,
	})

	data := LiveToVodResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return &data, err
}
