package channel_audits

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
)

// number of channel audits to request at once
const pageSize = 100

const (
	ActionRemoveEditor  = "remove_editor"
	ActionAddEditor     = "add_editor"
	ActionRunCommercial = "commercial"
	ActionGameChange    = "game_change"
	ActionStatusChange  = "status_change"
)

type Client interface {
	GetChannelAudits(ctx context.Context, params GetChannelAuditsParams, reqOpts *twitchclient.ReqOpts) (*ChannelAuditsResult, error)
	CreateChannelAudit(ctx context.Context, params CreateChannelAuditParams, reqOpts *twitchclient.ReqOpts) error
}

type clientImpl struct {
	twitchclient.Client
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = "channel_audits"
	}
	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &clientImpl{twitchClient}, nil
}

type CreateChannelAuditParams struct {
	ActorID   string  `json:"actor_id"`            // ActorID is the ID of the user making the change
	ChannelID string  `json:"obj_id"`              // ChannelID is the channel being modified
	OldValue  *string `json:"old_value,omitempty"` // OldValue is the old
	NewValue  *string `json:"new_value,omitempty"` // NewValue is the value after the update
	Action    string  `json:"action"`              // Action is the action being performed.
}

func (c *clientImpl) CreateChannelAudit(ctx context.Context, params CreateChannelAuditParams, reqOpts *twitchclient.ReqOpts) (err error) {
	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.channel_audits.create_channel_audit",
		StatSampleRate: 0.1,
	})
	body, err := json.Marshal(params)
	if err != nil {
		return err
	}
	path := fmt.Sprintf("/v1/channel_audits/%s", params.ChannelID)
	req, err := c.NewRequest(http.MethodPost, path, bytes.NewReader(body))
	if err != nil {
		return err
	}
	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		cErr := resp.Body.Close()
		if err == nil {
			err = cErr
		}
	}()
	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}
	return nil
}

type GetChannelAuditsParams struct {
	Channel string   // ID of channel you are interested in
	Limit   int      // maximum number of channel audits to return, < 0 for no limit
	After   int64    // time (in seconds since 1970), only return channel audits after this time
	Before  int64    // '' before this time
	Actions []string // only return CAs for listed actions, empty to not filter, actions:['commercial', 'cut', 'status_change', 'game_change', 'add_editor', 'remove_editor']
}

func (c *clientImpl) GetChannelAudits(ctx context.Context, params GetChannelAuditsParams, reqOpts *twitchclient.ReqOpts) (*ChannelAuditsResult, error) {
	var cursor int64 = 0
	var finalResult ChannelAuditsResult
	limit := params.Limit

	// Grab up to pageSize results, grab more if pageSize was reached
	for got := pageSize; got == pageSize && limit != 0; {
		//build request string
		b := bytes.Buffer{}
		b.Write([]byte(fmt.Sprintf("/v1/channel_audits/%s?after=%d&before=%d&limit=%d", params.Channel, params.After, params.Before, toGet(limit, pageSize))))
		if cursor > 0 {
			b.Write([]byte(fmt.Sprintf("&cursor=%d", cursor)))
		}
		for _, action := range params.Actions {
			b.Write([]byte(fmt.Sprintf("&actions[]=%s", action)))
		}

		// issue request to channel-audits API
		req, err := c.NewRequest("GET", b.String(), nil)
		if err != nil {
			return nil, err
		}

		combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
			StatName:       "service.channel_audits.channel_audits",
			StatSampleRate: 0.1,
		})

		var nextResult []ChannelAudit
		_, err = c.DoJSON(ctx, &nextResult, req, combinedReqOpts)
		if err != nil {
			return nil, err
		}

		// add result, check number of channel audits returned
		got = len(nextResult)
		if got > 0 {
			finalResult.ChannelAudits = append(finalResult.ChannelAudits, nextResult...)
			cursor = nextResult[got-1].ID
			limit -= got // a positive limit will never become negative, since we'll never get more than limit
		}
	}
	return &finalResult, nil
}

// grab pgSiz results, unless a nonnegative limit is smaller
func toGet(lim, pgSiz int) int {
	if lim >= 0 && lim < pgSiz {
		return lim
	}
	return pgSiz
}
