// +build integration

package main

import (
	"fmt"
	"math/rand"
	"net"
	"net/http"
	"net/url"
	"os"
	"strconv"
	"strings"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/chat/friendship/client"
	"code.justin.tv/discovery/recommendations/client"
	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/ranker"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/recommendations/mocks"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	cohesion "code.justin.tv/web/cohesion/client/v2"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

func timestamp() string {
	return time.Now().Format("2006-01-02-15.04.05.000000000-MST")
}

func limit(i int) *int {
	return &i
}

func getFeed(ts *testSetup, feedID string, limit *int, cursor *string) (*models.Feed, error) {
	hostPath := fmt.Sprintf("/v1/feeds/%s", feedID)
	queryParams := make(url.Values)
	if limit != nil {
		queryParams.Add("limit", fmt.Sprintf("%d", *limit))
	}
	if cursor != nil {
		queryParams.Add("cursor", *cursor)
	}
	var feed models.Feed
	if err := clients.DoHTTP(ts.ctx, ts.client, "GET", ts.host+hostPath, queryParams, nil, &feed, nil); err != nil {
		return nil, err
	}
	return &feed, nil
}

func saveStory(ts *testSetup, feedID string, storyID string, actor entity.Entity, verb verb.Verb, entity entity.Entity, score float64) (*models.Story, error) {
	url := "/private/story"
	story := models.Story{
		FeedID:  feedID,
		StoryID: storyID,
		Score:   score,
		Activity: models.Activity{
			Entity: entity,
			Actor:  actor,
			Verb:   verb,
		},
	}
	var ret models.Story
	if err := clients.DoHTTP(ts.ctx, ts.client, "POST", ts.host+url, nil, story, &ret, nil); err != nil {
		return nil, err
	}
	return &ret, nil
}

func removeStory(ts *testSetup, feedID, storyID string) error {
	url := fmt.Sprintf("/v1/story/%s/%s", feedID, storyID)
	var ret string
	if err := clients.DoHTTP(ts.ctx, ts.client, "DELETE", ts.host+url, nil, nil, &ret, nil); err != nil {
		return err
	}
	return nil
}

func TestIntegration_TraitLoader(t *testing.T) {
	t.Parallel()
	Convey("With loader", t, func() {
		localConf := &distconf.InMemory{}
		err := addMapValues(localConf, map[string][]byte{
			"masonry.listen_addr":                    []byte(":0"),
			"rollbar.access_token":                   []byte(""),
			"statsd.hostport":                        []byte(""),
			"debug.addr":                             []byte(":0"),
			"logging.to_stdout":                      []byte("false"),
			"masonry.low.sqssource.draining_threads": []byte("0"),
			"masonry.mid.sqssource.draining_threads": []byte("0"),
			"masonry.use_new_ranker":                 []byte("0.0"),
		})
		So(err, ShouldBeNil)

		configCommon := service_common.ConfigCommon{
			Team:          teamName,
			Service:       serviceName,
			CustomReaders: []distconf.Reader{localConf},
			BaseDirectory: "../../",
		}
		So(configCommon.Setup(), ShouldBeNil)
		cohesionFollows, err := cohesion.New(configCommon.Config.Str("candidatediscovery.follows.cohesion_rpc_url", "").Get(), "follows")
		So(err, ShouldBeNil)

		friendshipClient, err := friendship.NewClient(twitchclient.ClientConf{Host: configCommon.Config.Str("candidatediscovery.friends.friendship_url", "").Get()})
		So(err, ShouldBeNil)

		l := ranker.ActorRelationshipTraitLoader{
			Cohesion:         cohesionFollows,
			FriendshipClient: friendshipClient,
			Rand:             rand.New(rand.NewSource(0)),
			SkipRelationshipLoading: configCommon.Config.Float("skip_relationships", 0.0),
			MaxSplitSize:            configCommon.Config.Int("split_size", 100),
			MetadataCacheValidTill:  configCommon.Config.Duration("valid_till", time.Second*15),
			Stats: service_common.NopStatSender(),
		}

		duploConfig := duplo.Config{}
		So(duploConfig.Load(configCommon.Config), ShouldBeNil)

		duploClient := &duplo.Client{
			Config: &duploConfig,
		}

		serialLoader := ranker.SerialTraitLoader{
			ActorRelationshipTraitLoader: l,
			ActorTraitLoader:             ranker.ActorTraitLoader{},
			EntityTraitLoader: ranker.EntityTraitLoader{
				DuploClient:  duploClient,
				Stats:        service_common.NopStatSender(),
				OldDataValid: configCommon.Config.Duration("...", time.Second),
			},
			FeedTraitLoader: ranker.FeedTraitLoader{},
			Stats:           service_common.NopStatSender(),
		}

		ctx := context.Background()
		actorFollower := entity.New("user", "27184041")
		actorFollowed := entity.New("user", "27222385")
		actorNobody := entity.New("user", "27222384")
		err = cohesionFollows.Create(
			ctx,
			cohesion.Entity{ID: actorFollower.ID(), Kind: "user"},
			"follows",
			cohesion.Entity{ID: actorFollowed.ID(), Kind: "user"},
			nil,
		)
		if err != nil && !strings.Contains(err.Error(), "already exists") {
			t.Error("Invalid error from associations")
		}

		So(cohesionFollows.Delete(
			ctx,
			cohesion.Entity{ID: actorFollowed.ID(), Kind: "user"},
			"follows",
			cohesion.Entity{ID: actorFollower.ID(), Kind: "user"},
		), ShouldBeNil)
		Convey("Follows should work both direction", func() {
			resp, err := cohesionFollows.Get(ctx, cohesion.Entity{ID: actorFollower.ID(), Kind: "user"},
				"follows",
				cohesion.Entity{ID: actorFollowed.ID(), Kind: "user"})
			So(err, ShouldBeNil)
			So(len(resp.Associations), ShouldBeGreaterThan, 0)

			resp, err = cohesionFollows.Get(ctx, cohesion.Entity{ID: actorFollowed.ID(), Kind: "user"},
				"followed_by",
				cohesion.Entity{ID: actorFollower.ID(), Kind: "user"})
			So(err, ShouldBeNil)
			So(len(resp.Associations), ShouldBeGreaterThan, 0)

			resp, err = cohesionFollows.Get(ctx, cohesion.Entity{ID: actorFollower.ID(), Kind: "user"},
				"followed_by",
				cohesion.Entity{ID: actorFollowed.ID(), Kind: "user"})
			So(err, ShouldBeNil)
			So(resp, ShouldBeNil)

			resp, err = cohesionFollows.Get(ctx, cohesion.Entity{ID: actorFollowed.ID(), Kind: "user"},
				"follows",
				cohesion.Entity{ID: actorFollower.ID(), Kind: "user"})
			So(err, ShouldBeNil)
			So(resp, ShouldBeNil)

		})
		Convey("Should not see follow wrong way", func() {
			multiTraits, err := l.ForMultipleToActors(ctx, actorFollowed, []entity.Entity{actorFollower}, nil)
			So(err, ShouldBeNil)
			So(multiTraits[0].Follows, ShouldBeFalse)
			So(multiTraits[0].Friends, ShouldBeFalse)

			Convey("But should respect skip loading relationships", func() {
				localConf.Write("skip_relationships", []byte("1.0"))
				multiTraits, err := l.ForMultipleToActors(ctx, actorFollowed, []entity.Entity{actorFollower}, nil)
				So(err, ShouldBeNil)
				So(multiTraits[0].Follows, ShouldBeTrue)
				So(multiTraits[0].Friends, ShouldBeTrue)
			})
		})
		Convey("Should see follow", func() {

			// Set it backwards so it shouldn't be used
			m := feeddataflow.Metadata{}
			m.SetFollow(actorFollowed.ID(), actorFollower.ID(), false, time.Now())

			multiTraits, err := l.ForMultipleToActors(ctx, actorFollower, []entity.Entity{actorFollowed}, nil)
			So(err, ShouldBeNil)
			So(multiTraits[0].Follows, ShouldBeTrue)
			So(multiTraits[0].Friends, ShouldBeFalse)
		})

		Convey("Metadata should load", func() {
			m := feeddataflow.Metadata{}
			m.SetFollow(actorFollower.ID(), actorFollowed.ID(), false, time.Now())
			multiTraits, err := l.ForMultipleToActors(ctx, actorFollower, []entity.Entity{actorFollowed}, &m)
			So(err, ShouldBeNil)
			So(multiTraits[0].Follows, ShouldBeFalse)
			So(multiTraits[0].Friends, ShouldBeFalse)
		})

		Convey("Multiget should work", func() {
			traitsMulti, err := l.ForMultipleToActors(ctx, actorFollower, []entity.Entity{actorFollowed, actorNobody, actorFollower}, nil)
			So(err, ShouldBeNil)
			So(len(traitsMulti), ShouldEqual, 3)
			So(traitsMulti[0].Follows, ShouldBeTrue)
			So(traitsMulti[0].Friends, ShouldBeFalse)

			So(traitsMulti[1].Follows, ShouldBeFalse)
			So(traitsMulti[1].Friends, ShouldBeFalse)

			So(traitsMulti[2].Follows, ShouldBeFalse)
			So(traitsMulti[2].Friends, ShouldBeFalse)

			traitsMulti, err = l.ForMultipleToActors(ctx, actorFollowed, []entity.Entity{actorFollower, actorNobody}, nil)
			So(err, ShouldBeNil)
			So(len(traitsMulti), ShouldEqual, 2)
			So(traitsMulti[0].Follows, ShouldBeFalse)
			So(traitsMulti[0].Friends, ShouldBeFalse)

			So(traitsMulti[1].Follows, ShouldBeFalse)
			So(traitsMulti[1].Friends, ShouldBeFalse)
		})

		Convey("Story traits should load correctly", func() {
			post1, err := duploClient.CreatePost(ctx, actorFollowed.ID(), "hello world", nil)
			So(err, ShouldBeNil)

			post2, err := duploClient.CreatePost(ctx, actorFollowed.ID(), "hello world", nil)
			So(err, ShouldBeNil)

			post3, err := duploClient.CreatePost(ctx, actorFollower.ID(), "hello world", nil)
			So(err, ShouldBeNil)

			Convey("Should work with more stories than feeds", func() {
				sb := &ranker.StoryBatch{
					FeedIDs: []string{"n:" + actorFollower.ID(), "n:" + actorFollowed.ID()},
					Stories: []*ranker.Story{
						{
							Activity: &ranker.Activity{
								Entity: entity.New(entity.NamespacePost, post1.ID),
								Verb:   verb.Create,
								Actor:  actorFollowed,
							},
							StoryID: post1.ID,
						},
						{
							Activity: &ranker.Activity{
								Entity: entity.New(entity.NamespacePost, post2.ID),
								Verb:   verb.Create,
								Actor:  actorFollowed,
							},
							StoryID: post2.ID,
						},
						{
							Activity: &ranker.Activity{
								Entity: entity.New(entity.NamespacePost, post3.ID),
								Verb:   verb.Create,
								Actor:  actorFollower,
							},
							StoryID: post3.ID,
						},
					},
				}
				batchTraits, err := serialLoader.LoadTraits(ctx, sb)
				So(err, ShouldBeNil)
				So(batchTraits, ShouldNotBeNil)

				So(batchTraits.FeedTraits["n:"+actorFollowed.ID()].Owner.ID(), ShouldEqual, actorFollowed.ID())

				// TODO: We need consistent reads before we can do this
				//So(batchTraits.EntityTraits[post1.ID].CreationTime, ShouldEqual, post1.CreatedAt)

				_, exists := batchTraits.ActorRelationshipTraits[actorFollowed]
				So(exists, ShouldBeTrue)

				So(batchTraits.ActorRelationshipTraits[actorFollower][actorFollowed].Follows, ShouldBeTrue)
				So(batchTraits.ActorRelationshipTraits[actorFollower][actorFollowed].Friends, ShouldBeFalse)

				So(batchTraits.ActorRelationshipTraits[actorFollowed][actorFollower].Follows, ShouldBeFalse)
				So(batchTraits.ActorRelationshipTraits[actorFollowed][actorFollower].Friends, ShouldBeFalse)
			})
			Convey("Should work with more feeds than stories", func() {
				sb := &ranker.StoryBatch{
					FeedIDs: []string{"n:" + actorFollower.ID(), "n:" + actorFollowed.ID(), "c:" + actorFollower.ID(), "c:" + actorFollowed.ID()},
					Stories: []*ranker.Story{
						{
							Activity: &ranker.Activity{
								Entity: entity.New(entity.NamespacePost, post1.ID),
								Verb:   verb.Create,
								Actor:  actorFollowed,
							},
							StoryID: post1.ID,
						},
						{
							Activity: &ranker.Activity{
								Entity: entity.New(entity.NamespacePost, post2.ID),
								Verb:   verb.Create,
								Actor:  actorFollowed,
							},
							StoryID: post2.ID,
						},
						{
							Activity: &ranker.Activity{
								Entity: entity.New(entity.NamespacePost, post3.ID),
								Verb:   verb.Create,
								Actor:  actorFollower,
							},
							StoryID: post3.ID,
						},
					},
				}
				batchTraits, err := serialLoader.LoadTraits(ctx, sb)
				So(err, ShouldBeNil)
				So(batchTraits, ShouldNotBeNil)

				So(batchTraits.FeedTraits["n:"+actorFollowed.ID()].Owner.ID(), ShouldEqual, actorFollowed.ID())

				// TODO: We need consistent reads before we can do this
				//So(batchTraits.EntityTraits[post1.ID].CreationTime, ShouldEqual, post1.CreatedAt)

				_, exists := batchTraits.ActorRelationshipTraits[actorFollowed]
				So(exists, ShouldBeTrue)

				So(batchTraits.ActorRelationshipTraits[actorFollower][actorFollowed].Follows, ShouldBeTrue)
				So(batchTraits.ActorRelationshipTraits[actorFollower][actorFollowed].Friends, ShouldBeFalse)

				So(batchTraits.ActorRelationshipTraits[actorFollowed][actorFollower].Follows, ShouldBeFalse)
				So(batchTraits.ActorRelationshipTraits[actorFollowed][actorFollower].Friends, ShouldBeFalse)
			})
		})
	})
}

func assertCanFetchEmptyFeed(ts *testSetup, feedID string) {
	feed, err := getFeed(ts, feedID, nil, nil)
	So(err, ShouldBeNil)
	So(feed, ShouldNotBeNil)
	So(feed.ID, ShouldEqual, feedID)
	So(feed.Items, ShouldBeEmpty)
	So(feed.Cursor, ShouldBeBlank)
}

func assertCanSaveAndDeleteStoryInFeed(ts *testSetup, feedID string, storyID string, a entity.Entity, v verb.Verb, e entity.Entity, score float64) {
	story, err := saveStory(ts, feedID, storyID, a, v, e, score)
	So(err, ShouldBeNil)
	So(story, ShouldNotBeNil)
	So(story.StoryID, ShouldEqual, storyID)

	feed, err := getFeed(ts, feedID, nil, nil)
	So(err, ShouldBeNil)
	So(feed, ShouldNotBeNil)
	So(feed.ID, ShouldEqual, feedID)
	So(feed.Items, ShouldHaveLength, 1)
	So(*feed.Items[0], ShouldResemble, story.Activity)

	err = removeStory(ts, feedID, storyID)
	So(err, ShouldBeNil)

	feed, err = getFeed(ts, feedID, nil, nil)
	So(err, ShouldBeNil)
	So(feed, ShouldNotBeNil)
	So(feed.ID, ShouldEqual, feedID)
	So(feed.Items, ShouldHaveLength, 0)
}

func TestIntegration_ChannelFeed(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)

		Convey("Should be able to get an empty feed", func() {
			testFeedID := "c:" + timestamp()
			assertCanFetchEmptyFeed(ts, testFeedID)
		})

		Convey("Should be able to save and deleted a story in the feed", func() {
			testFeedID := "c:" + timestamp()
			assertCanSaveAndDeleteStoryInFeed(ts, testFeedID, "s:1", entity.New("a", "1"), verb.Create, entity.New("e", "1"), 1.0)
		})

		Convey("Should be able to save two stories and paginate the feed", func() {
			testFeedID := "c:" + timestamp()
			story1, err := saveStory(ts, testFeedID, "s:1", entity.New("a", "1"), verb.Create, entity.New("e", "1"), 1.0)
			So(err, ShouldBeNil)
			So(story1, ShouldNotBeNil)
			So(story1.StoryID, ShouldEqual, "s:1")

			story2, err := saveStory(ts, testFeedID, "s:2", entity.New("a", "1"), verb.Create, entity.New("e", "2"), 2.0)
			So(err, ShouldBeNil)
			So(story2, ShouldNotBeNil)
			So(story2.StoryID, ShouldEqual, "s:2")

			feed1, err := getFeed(ts, testFeedID, limit(1), nil)
			So(err, ShouldBeNil)
			So(feed1, ShouldNotBeNil)
			So(feed1.Cursor, ShouldNotBeBlank)
			So(feed1.Items, ShouldHaveLength, 1)
			So(*feed1.Items[0], ShouldResemble, story2.Activity)

			feed2, err := getFeed(ts, testFeedID, limit(1), &feed1.Cursor)
			So(err, ShouldBeNil)
			So(feed2, ShouldNotBeNil)
			So(feed2.Cursor, ShouldBeBlank)
			So(feed2.Items, ShouldHaveLength, 1)
			So(*feed2.Items[0], ShouldResemble, story1.Activity)
		})

		Convey("When paging through a channel feed", func() {
			feedID := "c:" + timestamp()
			_, err := saveStory(ts, feedID, "s:1", entity.New("a", "1"), verb.Create, entity.New("e", "1"), 1.0)
			So(err, ShouldBeNil)

			_, err = saveStory(ts, feedID, "s:2", entity.New("a", "1"), verb.Create, entity.New("e", "2"), 2.0)
			So(err, ShouldBeNil)

			feed, err := getFeed(ts, feedID, limit(1), nil)
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(feed.Cursor, ShouldNotBeBlank)
			So(feed.Tracking, ShouldNotBeNil)

			batchID := feed.Tracking.BatchID
			So(batchID, ShouldNotBeEmpty)

			Convey("Requesting the next page should also return the same batch ID", func() {
				feed, err = getFeed(ts, feedID, limit(1), &feed.Cursor)
				So(err, ShouldBeNil)
				So(feed, ShouldNotBeNil)
				So(feed.Tracking, ShouldNotBeNil)
				So(feed.Tracking.BatchID, ShouldEqual, batchID)
			})

			Convey("Reloading the feed should return a new batch ID", func() {
				feed, err = getFeed(ts, feedID, limit(1), nil)
				So(err, ShouldBeNil)
				So(feed, ShouldNotBeNil)
				So(feed.Tracking, ShouldNotBeNil)
				So(feed.Tracking.BatchID, ShouldNotEqual, batchID)
			})
		})
	})
}

func TestIntegration_NewsFeed(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)

		Convey("Should be able to get an empty feed", func() {
			testFeedID := "n:" + timestamp()
			assertCanFetchEmptyFeed(ts, testFeedID)
		})

		Convey("Should be able to save and deleted a story in the feed", func() {
			testFeedID := "n:" + timestamp()
			assertCanSaveAndDeleteStoryInFeed(ts, testFeedID, "s:1", entity.New("a", "1"), verb.Create, entity.New("e", "1"), 1.0)
		})

		Convey("Should be able to save three stories and paginate the feed", func() {
			testFeedID := "n:" + timestamp()
			story1, err := saveStory(ts, testFeedID, "s:1", entity.New("a", "1"), verb.Create, entity.New("e", "1"), 1.0)
			So(err, ShouldBeNil)
			So(story1, ShouldNotBeNil)
			So(story1.StoryID, ShouldEqual, "s:1")

			story2, err := saveStory(ts, testFeedID, "s:2", entity.New("a", "1"), verb.Create, entity.New("e", "2"), 2.0)
			So(err, ShouldBeNil)
			So(story2, ShouldNotBeNil)
			So(story2.StoryID, ShouldEqual, "s:2")

			story3, err := saveStory(ts, testFeedID, "s:3", entity.New("a", "1"), verb.Create, entity.New("e", "2"), 3.0)
			So(err, ShouldBeNil)
			So(story3, ShouldNotBeNil)
			So(story3.StoryID, ShouldEqual, "s:3")

			feed1, err := getFeed(ts, testFeedID, limit(1), nil)
			So(err, ShouldBeNil)
			So(feed1, ShouldNotBeNil)
			So(feed1.Cursor, ShouldNotBeBlank)
			So(feed1.Items, ShouldHaveLength, 1)
			So(*feed1.Items[0], ShouldResemble, story3.Activity)

			feed2, err := getFeed(ts, testFeedID, limit(2), &feed1.Cursor)
			So(err, ShouldBeNil)
			So(feed2, ShouldNotBeNil)
			So(feed2.Cursor, ShouldBeBlank)
			So(feed2.Items, ShouldHaveLength, 2)
			So(*feed2.Items[0], ShouldResemble, story2.Activity)
			So(*feed2.Items[1], ShouldResemble, story1.Activity)
		})
	})
}

func TestIntegration_RecommendationFeed(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)

		Convey("Should be able to load recommended feeds", func() {
			testUserID := timestamp()
			limit := 2
			// This works because recommendations returns an anonymous list of suggestions if it can't find the user ID
			feed, err := getFeed(ts, "r:"+testUserID, &limit, nil)
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)

			So(feed.Tracking, ShouldNotBeNil)
			So(feed.Tracking.BatchID, ShouldNotBeBlank)
			So(feed.Items, ShouldNotBeNil)
			So(feed.Items, ShouldHaveLength, limit)

			item := feed.Items[0]
			So(item, ShouldNotBeNil)
			So(item.RelevanceReason, ShouldNotBeNil)
			So(item.RelevanceReason.Kind, ShouldNotEqual, "")
			So(item.RelevanceReason.Source, ShouldNotEqual, "")
			So(item.RecGenerationID, ShouldNotEqual, "")
			So(item.RecGenerationIndex, ShouldNotBeNil)
			So(*item.RecGenerationIndex, ShouldEqual, 0)
		})

		Convey("After a recommended feed is loaded", func() {
			testUserID := timestamp()
			limit := 1
			feedID := "r:" + testUserID

			// This works because recommendations returns an anonymous list of suggestions if it can't find the user ID
			feed, err := getFeed(ts, feedID, &limit, nil)

			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
			So(feed.Cursor, ShouldNotBeBlank)
			So(feed.Tracking, ShouldNotBeNil)
			So(feed.Tracking.BatchID, ShouldNotBeBlank)

			batchID := feed.Tracking.BatchID

			Convey("Requesting the next page should also return the same batch ID", func() {
				feed, err = getFeed(ts, feedID, &limit, &feed.Cursor)
				So(err, ShouldBeNil)
				So(feed, ShouldNotBeNil)
				So(feed.Tracking, ShouldNotBeNil)
				So(feed.Tracking.BatchID, ShouldEqual, batchID)
			})

			Convey("Reloading the feed should return a new batch ID", func() {
				feed, err = getFeed(ts, feedID, &limit, nil)
				So(err, ShouldBeNil)
				So(feed, ShouldNotBeNil)
				So(feed.Tracking, ShouldNotBeNil)
				So(feed.Tracking.BatchID, ShouldNotEqual, batchID)
			})
		})
	})
}

func TestIntegration_RecommendationsFeed_WithMockClient(t *testing.T) {
	t.Parallel()
	mockRecClient := &mocks.Client{}

	ts := startServer(t, injectables{RecommendationsClient: mockRecClient}, map[string][]byte{
		"masonry.recs-load-more-at": []byte("-1"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	rec := func(kind, id string) *recommendations.Recommendation {
		ret := recommendations.Recommendation{
			Kind: kind,
			ID:   id,
		}
		return &ret
	}

	mockGetRecommendations := func(recs ...*recommendations.Recommendation) {
		ret := &recommendations.Recommendations{
			GenerationID:    "test-" + timestamp(),
			Recommendations: recs,
		}
		mockRecClient.On("GetRecommendations", mock.Anything, mock.Anything, mock.Anything).Once().Return(ret, nil)
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		resetMock(&mockRecClient.Mock)

		Convey("Should be able to page through recommendations feed", func() {
			userID := strconv.FormatInt(time.Now().UnixNano(), 10)
			testFeedID := "r:" + userID

			mockGetRecommendations(rec("vod", "1"), rec("clip", "a"))

			// This should fetch the first recommendation
			f1, err := getFeed(ts, testFeedID, limit(1), nil)
			So(err, ShouldBeNil)
			So(f1, ShouldNotBeNil)
			So(f1.Cursor, ShouldNotBeBlank)
			So(f1.Items, ShouldHaveLength, 1)
			So(f1.Items[0].Entity, ShouldResemble, entity.New("vod", "1"))

			// Stub out the rest of the GetRecommendations calls
			mockRecClient.On("GetRecommendations", mock.Anything, mock.Anything, mock.Anything).Return(&recommendations.Recommendations{}, nil)

			// This should fetch the 2nd recommendation and end
			f2, err := getFeed(ts, testFeedID, limit(1), &f1.Cursor)
			So(err, ShouldBeNil)
			So(f2, ShouldNotBeNil)
			So(f2.Cursor, ShouldBeBlank)
			So(f2.Items, ShouldHaveLength, 1)
			So(f2.Items[0].Entity, ShouldResemble, entity.New("clip", "a"))
			So(f2.Tracking, ShouldNotBeNil)
			batchID := f2.Tracking.BatchID
			So(batchID, ShouldNotBeEmpty)
		})
	})
}

type httpError struct {
	statusCode int
	message    string
}

func (e httpError) Error() string {
	return fmt.Sprintf("%d: %s", e.statusCode, e.message)
}

func resetMock(m *mock.Mock) {
	m.ExpectedCalls = nil
	m.Calls = nil
}

type testSetup struct {
	ctx          context.Context
	client       *http.Client
	host         string
	onFinish     func(timeToWait time.Duration)
	thisInstance *service
}

func (t *testSetup) Setup() error {
	ctx, cancelFunc := context.WithTimeout(context.Background(), time.Second*5)
	t.ctx = ctx
	Reset(cancelFunc)
	t.client = &http.Client{}
	return nil
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func startServer(t *testing.T, i injectables, configs ...map[string][]byte) *testSetup {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"masonry.listen_addr":                    []byte(":0"),
		"rollbar.access_token":                   []byte(""),
		"statsd.hostport":                        []byte(""),
		"debug.addr":                             []byte(":0"),
		"logging.to_stdout":                      []byte("false"),
		"logging.to_stderr":                      []byte("true"),
		"masonry.low.sqssource.draining_threads": []byte("0"),
		"masonry.mid.sqssource.draining_threads": []byte("0"),
		"masonry.skip_trait_loading":             []byte("1.0"),
	})
	if err != nil {
		t.Error(err)
		return nil
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			t.Error(err)
			return nil
		}
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				ElevateLogKey: elevateKey,
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(vals ...interface{}) bool {
					return false
				},
			},
			SfxSetupConfig: sfxStastdConfig(),
			PanicLogger:    panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 20):
		t.Error("Took to long to start service")
		return nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testSetup{
		host:         addressForIntegrationTests,
		onFinish:     onFinish,
		thisInstance: &thisInstance,
	}
}

func TestMainFunc(t *testing.T) {
	ret := 0
	a := instance.osExit
	b := instance.serviceCommon.BaseDirectory
	c := instance.serviceCommon.Log
	defer func() {
		instance.osExit = a
		instance.serviceCommon.BaseDirectory = b
		instance.serviceCommon.Log = c
	}()
	instance.osExit = func(i int) { ret = i }
	instance.serviceCommon.BaseDirectory = "asdfjadsfjsdkfjsdkjghkdls"
	instance.serviceCommon.Log = nil
	main()
	if ret == 0 {
		t.Error("expected instance to fail")
	}
}
