package masonry

import (
	"time"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/providers"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/ranker"
	"code.justin.tv/feeds/service-common"
	"golang.org/x/net/context"
)

// BatchProcessor accepts story batches and forwards them to storage
type BatchProcessor struct {
	Ranker  ranker.Ranker
	FeedMux *providers.FeedMux
	Log     log.Logger
	Backoff *service_common.ThrottledBackoff
}

// StoryStorage stores a scored story
type StoryStorage interface {
	SaveStory(ctx context.Context, feedID, storyID string, activity models.Activity, score float64, expiresAt *time.Time) (*models.Story, error)
	RemoveStory(ctx context.Context, feedID, storyID string) error
}

// ToRankerActivityBatch converts between masonry and ranker activity batches
func ToRankerActivityBatch(s *models.ActivityBatch) *ranker.ActivityBatch {
	ret := ranker.ActivityBatch{
		FeedIDs:    s.FeedIDs,
		Activities: make([]*ranker.Activity, 0, len(s.Activities)),
		Metadata:   &feeddataflow.Metadata{},
	}
	ret.Metadata.MergeFrom(s.Metadata)
	for _, a := range s.Activities {
		ret.Metadata.MergeFrom(a.Metadata)
		ret.Activities = append(ret.Activities, &ranker.Activity{
			Entity: a.Entity,
			Verb:   a.Verb,
			Actor:  a.Actor,
		})
	}
	return &ret
}

// ProcessStoryBatch processes the story batch
func (b *BatchProcessor) ProcessStoryBatch(ctx context.Context, sb *models.ActivityBatch) error {
	s := ToRankerActivityBatch(sb)
	scores, err := b.Ranker.Score(ctx, s)
	if err != nil {
		b.Log.Log("err", err, "unable to score a story")
		return err
	}
	for _, feedID := range s.FeedIDs {
		for _, activity := range s.Activities {
			storyID, score := scores.Rank(activity, feedID)
			if storyID == "" {
				continue
			}
			canSaveActivity := models.Activity{
				Entity: activity.Entity,
				Verb:   activity.Verb,
				Actor:  activity.Actor,
			}
			if err := b.throttleAndSave(ctx, feedID, canSaveActivity, storyID, score); err != nil {
				return err
			}
		}
	}
	return nil
}

func isThrottled(err error) bool {
	if err == nil {
		return false
	}
	type Throttled interface {
		IsThrottled() bool
	}
	throttledType, ok := errors.Cause(err).(Throttled)
	if !ok {
		return false
	}
	return throttledType.IsThrottled()
}

func (b *BatchProcessor) throttleAndSave(ctx context.Context, feedID string, activity models.Activity, storyID string, score float64) error {
	defer b.Backoff.DecreaseBackoff()
	for {
		if err := b.Backoff.ThrottledSleep(ctx); err != nil {
			return err
		}
		var err error
		if score == ranker.DeleteStoryScore {
			err = b.FeedMux.RemoveStory(ctx, feedID, storyID)
		} else {
			_, err = b.FeedMux.SaveStory(ctx, feedID, storyID, activity, score)
		}

		if !isThrottled(err) {
			return err
		}
		b.Log.Log("err", err, "throttled story")
		b.Backoff.SignalThrottled()
	}
}
