package masonry

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/providers"
	"code.justin.tv/feeds/service-common"
	"goji.io"
	"goji.io/pat"
)

// HTTPConfig configures HTTPServer
type HTTPConfig struct {
	service_common.BaseHTTPServerConfig
}

// Load load config values from distconf
func (c *HTTPConfig) Load(dconf *distconf.Distconf) error {
	return c.Verify(dconf, "masonry")
}

// HTTPServer answers masonry HTTP requests
type HTTPServer struct {
	service_common.BaseHTTPServer
	FeedMux              *providers.FeedMux
	PrivateBatchReciever BatchReciever
}

// SetupRoutes configures the http server
func (h *HTTPServer) SetupRoutes(mux *goji.Mux) {
	mux.Handle(pat.Get("/v1/feeds/:id"), h.createHandler("get_feed", h.getFeed))
	mux.Handle(pat.Delete("/v1/story/:feed_id/:story_id"), h.createHandler("remove_feed_story", h.removeFeedStory))
	mux.Handle(pat.Get("/debug/health"), h.createHandler("get_health", h.getHealth))

	mux.Handle(pat.Post("/private/story"), h.createHandler("test_post_story", h.privatePostStory))
	mux.Handle(pat.Post("/private/activity"), h.createHandler("test_post_activity_batch", h.privateActivityBatch))
}

func (h *HTTPServer) createHandler(name string, callback func(req *http.Request) (interface{}, error)) *service_common.JSONHandler {
	return &service_common.JSONHandler{
		Log:          h.Log,
		Stats:        h.Stats.NewSubStatSender(name),
		ItemCallback: callback,
	}
}

func parseInt(intParam string, defaultValue int64) (int64, error) {
	if intParam == "" {
		return defaultValue, nil
	}
	asInt, err := strconv.ParseInt(intParam, 10, 64)
	if err != nil {
		return 0, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, fmt.Sprintf("expect integer param, saw %s", intParam)),
		}
	}
	return asInt, nil
}

// GetFeedParams is a struct of query parameters that can be contained in a GetFeed request
type GetFeedParams struct {
	DeviceID string `json:"device_id"`
	Language string `json:"language"`
}

func (h *HTTPServer) getFeed(req *http.Request) (interface{}, error) {
	ctx := req.Context()

	feedID := pat.Param(req, "id")
	cursor := req.URL.Query().Get("cursor")

	params := &GetFeedParams{
		DeviceID: req.URL.Query().Get("device_id"),
		Language: req.URL.Query().Get("language"),
	}

	var err error
	var limit int64
	if limit, err = parseInt(req.URL.Query().Get("limit"), 10); err != nil {
		return nil, err
	}

	feed, err := h.FeedMux.GetFeed(ctx, feedID, limit, cursor, providers.GetFeedOptions{
		DeviceID: params.DeviceID,
		Language: params.Language,
	})

	if err != nil {
		return nil, errors.Wrap(err, "Cannot get feed items")
	}
	return feed, nil
}

func (h *HTTPServer) removeFeedStory(req *http.Request) (interface{}, error) {
	feedID := pat.Param(req, "feed_id")
	storyID := pat.Param(req, "story_id")
	err := h.FeedMux.RemoveStory(req.Context(), feedID, storyID)
	if err != nil {
		return nil, errors.Wrapf(err, "unable to remove feed story %s:%s", feedID, storyID)
	}
	return "", nil
}

func (h *HTTPServer) getHealth(req *http.Request) (interface{}, error) {
	return "OK", nil
}

func (h *HTTPServer) privatePostStory(req *http.Request) (interface{}, error) {
	var story models.Story
	if err := json.NewDecoder(req.Body).Decode(&story); err != nil {
		return nil, err
	}
	return h.FeedMux.SaveStory(req.Context(), story.FeedID, story.StoryID, story.Activity, story.Score)
}

func (h *HTTPServer) privateActivityBatch(req *http.Request) (interface{}, error) {
	var activityBatch models.ActivityBatch
	if err := json.NewDecoder(req.Body).Decode(&activityBatch); err != nil {
		return nil, err
	}
	return nil, h.PrivateBatchReciever.ProcessStoryBatch(req.Context(), &activityBatch)
}
