package masonry

import (
	"encoding/json"
	"strings"
	"time"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/service-common/feedsqs"
	"github.com/aws/aws-sdk-go/service/sqs"
	"golang.org/x/net/context"
)

// SQSHighPriorityInputConfig configures SqsSourceConstructor
type SQSHighPriorityInputConfig struct {
	feedsqs.SQSQueueProcessorConfig
}

// Load configuration information
func (c *SQSHighPriorityInputConfig) Load(dconf *distconf.Distconf) error {
	return c.SQSQueueProcessorConfig.Verify(dconf, "masonry.high", time.Second*3, 8, time.Minute*15)
}

// SQSMidPriorityInputConfig configures SqsSourceConstructor
type SQSMidPriorityInputConfig struct {
	feedsqs.SQSQueueProcessorConfig
}

// Load configuration information
func (c *SQSMidPriorityInputConfig) Load(dconf *distconf.Distconf) error {
	return c.SQSQueueProcessorConfig.Verify(dconf, "masonry.mid", time.Second*3, 16, time.Minute*15)
}

// SQSLowPriorityInputConfig configures SqsSourceConstructor
type SQSLowPriorityInputConfig struct {
	feedsqs.SQSQueueProcessorConfig
}

// Load configuration information
func (c *SQSLowPriorityInputConfig) Load(dconf *distconf.Distconf) error {
	return c.SQSQueueProcessorConfig.Verify(dconf, "masonry.low", time.Second*3, 6, time.Minute*15)
}

// SQSInput sends stories from SQS into a receiver
type SQSInput struct {
	feedsqs.SQSQueueProcessor
	Destination BatchReciever
}

// ComboQueueInput can process both old and new sqs items
type ComboQueueInput struct {
	Activities []*models.Activity `json:"activities"`
	FeedIDs    []string           `json:"feed_ids"`
	// An aggregate metadata for all the stories in the batch
	Metadata *feeddataflow.Metadata `json:"metadata,omitempty"`

	// Old traits.  For migration
	StoryID  string          `json:"story_id"`
	Activity models.Activity `json:"activity"`
}

func (s *SQSInput) processMessage(ctx context.Context, m *sqs.Message) error {
	msg := &ComboQueueInput{}
	err := json.NewDecoder(strings.NewReader(*m.Body)).Decode(msg)
	if err != nil {
		return err
	}
	s.Log.Debug("New valid message seen")

	sb := &models.ActivityBatch{
		Activities: msg.Activities,
		FeedIDs:    msg.FeedIDs,
		Metadata:   msg.Metadata,
	}
	if len(sb.Activities) == 0 {
		sb.Activities = append(sb.Activities, &msg.Activity)
	}
	return s.Destination.ProcessStoryBatch(ctx, sb)
}

// Setup sets up ProcessMessage and calls sub Setup()
func (s *SQSInput) Setup() error {
	s.SQSQueueProcessor.ProcessMessage = s.processMessage
	return s.SQSQueueProcessor.Setup()
}

// BatchReciever is anything that can receive stories
type BatchReciever interface {
	ProcessStoryBatch(ctx context.Context, s *models.ActivityBatch) error
}
