package providers

import (
	"context"
	"strings"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/storage"
	"code.justin.tv/feeds/service-common/feedcache"
)

type ChannelFeedProviderConfig struct {
	storage.FeedStorageConfig
	cachedEnabled *distconf.Bool
	cacheLimit    *distconf.Int
}

// Load load config values from distconf
func (c *ChannelFeedProviderConfig) Load(dconf *distconf.Distconf) error {
	c.cachedEnabled = dconf.Bool("masonry.channel_feed.cache", true)
	c.cacheLimit = dconf.Int("masonry.channel_feed.cache_limit", 10)
	return c.FeedStorageConfig.Verify(dconf, "channel")
}

type ChannelFeedProvider struct {
	Config      *ChannelFeedProviderConfig
	FeedStorage *storage.FeedStorage
	Cache       *feedcache.ObjectCache
	Log         log.Logger
}

var _ MutableFeedProvider = &ChannelFeedProvider{}

func ChannelFeedMatcher(feedID string) bool {
	return strings.HasPrefix(feedID, "c:")
}

func (p *ChannelFeedProvider) shouldCache(ctx context.Context, feedID string, limit int64, cursor string) bool {
	if !p.Config.cachedEnabled.Get() {
		return false
	}
	// We can only easily support caching a feed that has a specific set of options.  Otherwise cache invalidation
	// becomes more complicated.
	// Only cache for fetches that have no cursor and that are requesting a specific number of items
	return cursor == "" && limit == p.Config.cacheLimit.Get()
}

func (p *ChannelFeedProvider) cacheKey(s string) string {
	return "fc:" + s
}

func (p *ChannelFeedProvider) GetFeed(ctx context.Context, feedID string, limit int64, cursor string, opts GetFeedOptions) (*models.Feed, error) {
	if !p.shouldCache(ctx, feedID, limit, cursor) {
		return p.FeedStorage.GetFeed(ctx, feedID, limit, cursor)
	}

	var ret *models.Feed
	err := p.Cache.Cached(ctx, p.cacheKey(feedID), func() (interface{}, error) {
		return p.FeedStorage.GetFeed(ctx, feedID, limit, cursor)
	}, &ret)
	if err != nil {
		return nil, err
	}
	return ret, nil
}

func (p *ChannelFeedProvider) invalidateFeed(ctx context.Context, feedID string) {
	if err := p.Cache.Invalidate(ctx, p.cacheKey(feedID)); err != nil {
		p.Log.Log("err", err, "unable to invalidate redis cache")
	}
}

func (p *ChannelFeedProvider) SaveStory(ctx context.Context, feedID string, storyID string, activity models.Activity, score float64) (*models.Story, error) {
	p.invalidateFeed(ctx, feedID)
	return p.FeedStorage.SaveStory(ctx, feedID, storyID, activity, score, nil)
}

func (p *ChannelFeedProvider) RemoveStory(ctx context.Context, feedID, storyID string) error {
	p.invalidateFeed(ctx, feedID)
	return p.FeedStorage.RemoveStory(ctx, feedID, storyID)
}
