package providers

import (
	"context"
	"strings"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/recommendations"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/storage"
)

type NewsFeedProviderConfig struct {
	storage.FeedStorageConfig
}

// Load load config values from distconf
func (c *NewsFeedProviderConfig) Load(dconf *distconf.Distconf) error {
	return c.FeedStorageConfig.Verify(dconf, "news")
}

type NewsFeedProvider struct {
	Config      *NewsFeedProviderConfig
	FeedStorage *storage.FeedStorage
	Log         log.Logger
}

func NewsFeedMatcher(feedID string) bool {
	return strings.HasPrefix(feedID, "n:")
}

func (p *NewsFeedProvider) GetFeed(ctx context.Context, feedID string, limit int64, cursor string, opts GetFeedOptions) (*models.Feed, error) {
	return p.FeedStorage.GetFeed(ctx, feedID, limit, cursor)
}

func (p *NewsFeedProvider) SaveStory(ctx context.Context, feedID string, storyID string, activity models.Activity, score float64) (*models.Story, error) {
	return p.FeedStorage.SaveStory(ctx, feedID, storyID, activity, score, nil)
}

func (p *NewsFeedProvider) RemoveStory(ctx context.Context, feedID, storyID string) error {
	return p.FeedStorage.RemoveStory(ctx, feedID, storyID)
}

func (p *NewsFeedProvider) GetNewsfeedForRecommendations(ctx context.Context, userID string, limit int) ([]recommendations.Recommendation, error) {
	var recs []recommendations.Recommendation
	newsfeed, err := p.GetFeed(ctx, "n:"+userID, int64(limit), "", GetFeedOptions{})
	if err != nil {
		return nil, err
	}
	for _, post := range newsfeed.Items {
		recs = append(recs, recommendations.Recommendation{
			Type: post.Entity.Namespace(),
			ID:   post.Entity.ID(),
		})
	}
	return recs, nil
}
