package providers

import (
	"context"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
)

type GetFeedOptions struct {
	DeviceID string
	Language string
}

type FeedPattern func(feedID string) bool

type FeedProvider interface {
	GetFeed(ctx context.Context, feedID string, limit int64, cursor string, opts GetFeedOptions) (*models.Feed, error)
}

type FeedMutator interface {
	SaveStory(ctx context.Context, feedID string, storyID string, activity models.Activity, score float64) (*models.Story, error)
	RemoveStory(ctx context.Context, feedID, storyID string) error
}

type MutableFeedProvider interface {
	FeedProvider
	FeedMutator
}

type FeedMux struct {
	providers []feedMuxProvider
	mutators  []feedMuxMutator
	Log       log.Logger
}

var _ FeedProvider = &FeedMux{}
var _ MutableFeedProvider = &FeedMux{}

type feedMuxProvider struct {
	pattern  FeedPattern
	provider FeedProvider
}

type feedMuxMutator struct {
	pattern FeedPattern
	mutator FeedMutator
}

func (m *FeedMux) AddProvider(pattern FeedPattern, provider FeedProvider) {
	m.providers = append(m.providers, feedMuxProvider{
		pattern:  pattern,
		provider: provider,
	})
}

func (m *FeedMux) AddMutableProvider(pattern FeedPattern, provider MutableFeedProvider) {
	m.providers = append(m.providers, feedMuxProvider{
		pattern:  pattern,
		provider: provider,
	})
	m.mutators = append(m.mutators, feedMuxMutator{
		pattern: pattern,
		mutator: provider,
	})
}

func (m *FeedMux) getProvider(feedID string) (FeedProvider, error) {
	for _, provider := range m.providers {
		if provider.pattern(feedID) {
			return provider.provider, nil
		}
	}
	return nil, errors.Errorf("no provider found for feed id: %s", feedID)
}

func (m *FeedMux) GetFeed(ctx context.Context, feedID string, limit int64, cursor string, opts GetFeedOptions) (*models.Feed, error) {
	provider, err := m.getProvider(feedID)
	if err != nil {
		return nil, err
	}
	return provider.GetFeed(ctx, feedID, limit, cursor, opts)
}

func (m *FeedMux) getMutator(feedID string) (FeedMutator, error) {
	for _, mutator := range m.mutators {
		if mutator.pattern(feedID) {
			return mutator.mutator, nil
		}
	}
	return nil, errors.Errorf("no mutator found for feed id: %s", feedID)
}

func (m *FeedMux) SaveStory(ctx context.Context, feedID string, storyID string, activity models.Activity, score float64) (*models.Story, error) {
	mutator, err := m.getMutator(feedID)
	if err != nil {
		return nil, err
	}
	return mutator.SaveStory(ctx, feedID, storyID, activity, score)
}

func (m *FeedMux) RemoveStory(ctx context.Context, feedID string, storyID string) error {
	mutator, err := m.getMutator(feedID)
	if err != nil {
		return err
	}
	return mutator.RemoveStory(ctx, feedID, storyID)
}
