package providers

import (
	"context"
	"strings"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/recommendations"
)

const (
	// ReasonSourceRecommendations identifies relevance reasons that came from the recommendations service.
	ReasonSourceRecommendations = "recs"
)

// recServiceActor is a generic actor we use for recommended posts
var recServiceActor = entity.New("r", "0")

type RecommendationsFeedProviderConfig struct {
}

// Load load config values from distconf
func (c *RecommendationsFeedProviderConfig) Load(dconf *distconf.Distconf) error {
	return nil
}

type RecommendationsFeedProvider struct {
	Config    *RecommendationsFeedProviderConfig
	RecLoader *recommendations.RecommendationLoader
	Log       log.Logger
}

func RecommendationsFeedMatcher(feedID string) bool {
	return strings.HasPrefix(feedID, "r:")
}

func (p *RecommendationsFeedProvider) GetFeed(ctx context.Context, feedID string, limit int64, cursor string, opts GetFeedOptions) (*models.Feed, error) {
	parts := strings.Split(feedID, ":")
	if len(parts) != 2 {
		return nil, errors.New("unrecognized feed ID")
	}
	userID := parts[1]

	// Added to verify if problem of empty device-id is upstream or downstream.
	if opts.DeviceID == "" {
		p.Log.Log("user", userID, "feedID", feedID, "empty deviceID")
	}

	recs, err := p.RecLoader.GetMoreRecommendations(ctx, userID, limit, cursor, opts.DeviceID, opts.Language)
	if err != nil {
		return nil, err
	}
	activities := make([]*models.Activity, 0, len(recs.Recs))
	for _, r := range recs.Recs {
		ent, err := entity.Decode(r.StoryID())
		if err != nil {
			return nil, errors.Wrap(err, "unable to decode rec entity")
		}
		activity := models.Activity{
			Entity: ent,
			Verb:   verb.Create,
			// At this point, we don't actually know who created it and it's not worth looking up
			Actor: recServiceActor,
			RelevanceReason: &models.RelevanceReason{
				Kind:     r.Reason.Kind,
				EntityID: r.Reason.ContextID,
				Source:   ReasonSourceRecommendations,
			},
			RecGenerationID:    r.RecGenerationID,
			RecGenerationIndex: r.RecGenerationIndex,
		}
		activities = append(activities, &activity)
	}
	return &models.Feed{
		ID:     feedID,
		Cursor: recs.Cursor,
		Items:  activities,
		Tracking: &models.FeedTracking{
			BatchID: recs.BatchID,
		},
	}, nil
}
