package ranker

import (
	"math/rand"
	"sync"

	"time"

	"strings"

	"code.justin.tv/chat/friendship/app/api/responses"
	friendship "code.justin.tv/chat/friendship/client"
	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	cohesion "code.justin.tv/web/cohesion/client/v2"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"
)

// CohesionClient is used for follow relationships
type CohesionClient interface {
	BatchAssociations(context.Context, cohesion.Entity, string, []string) (*cohesion.ListAssocResponseWithMeta, error)
}

// FriendshipClient asserts what functions are needed from friendship.Client to allow mocking
type FriendshipClient interface {
	GetFriends(ctx context.Context, id string, params friendship.GetFriendsParams, reqOpts *twitchclient.ReqOpts) (*responses.FriendListResponse, error)
}

var _ FriendshipClient = friendship.Client(nil)
var _ CohesionClient = cohesion.Client(nil)

// ActorRelationshipTraitLoader loads traits about the relationship between two actors
type ActorRelationshipTraitLoader struct {
	Cohesion               CohesionClient
	FriendshipClient       FriendshipClient
	Stats                  *service_common.StatSender
	MetadataCacheValidTill *distconf.Duration

	// % chance [0.0 - 1.0] that we will skip loading relationship traits and instead assume all people are
	//                      friends/followers of each other
	mu                      sync.Mutex
	Rand                    *rand.Rand
	SkipRelationshipLoading *distconf.Float
	MaxSplitSize            *distconf.Int
}

func splitArray(actors []entity.Entity, maxSize int) [][]entity.Entity {
	ret := make([][]entity.Entity, 0, len(actors)/maxSize+1)
	for len(actors) > maxSize {
		var nextAppend []entity.Entity
		nextAppend, actors = actors[0:maxSize], actors[maxSize:]
		ret = append(ret, nextAppend)
	}
	if len(actors) != 0 {
		ret = append(ret, actors)
	}
	return ret
}

func isFollow(userFromID string, userToID string, isReverse bool, metadata *feeddataflow.Metadata, validTill time.Time) (bool, bool) {
	var follows, exists bool
	var validAt time.Time
	if userFromID == userToID {
		// People do not follow themselves
		return false, true
	}
	if isReverse {
		follows, validAt, exists = metadata.GetFollow(userToID, userFromID)
	} else {
		follows, validAt, exists = metadata.GetFollow(userFromID, userToID)
	}
	if !exists || validAt.Before(validTill) {
		return false, false
	}
	return follows, true
}

func (u *ActorRelationshipTraitLoader) followsMulti(ctx context.Context, userFromID string, toActors []entity.Entity, isReverse bool, metadata *feeddataflow.Metadata) ([]bool, error) {
	// Note: Cohesion can't handle more than 100 at a time.  We split into size of 100 before getting follows from cohesion
	//       It would be awesome if cohesion didn't have this limitation
	splits := splitArray(toActors, int(u.MaxSplitSize.Get()))
	ret := make([]bool, 0, len(toActors))
	for _, split := range splits {
		retBool, err := u.followsMultiImpl(ctx, userFromID, split, isReverse, metadata)
		if err != nil {
			return nil, err
		}
		ret = append(ret, retBool...)
	}
	return ret, nil
}

func fromAssocs(res *cohesion.ListAssocResponseWithMeta, ret []bool, isReverse bool, idsToIdx map[string]int) ([]bool, error) {
	if res == nil || len(res.Associations) == 0 {
		return ret, nil
	}
	for _, assoc := range res.Associations {
		retIdx, exists := idsToIdx[assoc.Assoc.Entity.ID]
		if !exists {
			return nil, errors.Errorf("unable to find userID %s", assoc.Assoc.Entity.ID)
		}
		if isReverse {
			if assoc.Kind != "followed_by" {
				continue
			}
		} else {
			if assoc.Kind != "follows" {
				continue
			}
		}

		ret[retIdx] = true
	}
	return ret, nil
}

func (u *ActorRelationshipTraitLoader) followsMultiImpl(ctx context.Context, userFromID string, toActors []entity.Entity, isReverse bool, metadata *feeddataflow.Metadata) ([]bool, error) {
	toIds := make([]string, 0, len(toActors))
	idsToIdx := make(map[string]int, len(toActors))
	ret := make([]bool, len(toActors))

	for i := 0; i < len(toActors); i++ {
		userToID := toActors[i].ID()
		isFollow, exists := isFollow(userFromID, userToID, isReverse, metadata, time.Now().Add(-u.MetadataCacheValidTill.Get()))
		if exists {
			ret[i] = isFollow
			continue
		}
		toIds = append(toIds, userToID)
		idsToIdx[userToID] = i
	}

	if len(toIds) == 0 {
		return ret, nil
	}
	// TODO: We cannot use following service because they do not have batch get follows
	// See https://git-aws.internal.justin.tv/feeds/following-service/issues/18
	startTime := time.Now()
	res, err := retryBatchAssociations(ctx, cohesion.Entity{ID: userFromID, Kind: "user"}, "user", toIds, u.Cohesion)
	if err != nil {
		return nil, errors.Wrapf(err, "unable to discover follow %s -> multi (total time=%s len(toIds)=%d)", userFromID, time.Since(startTime), len(toIds))
	}
	return fromAssocs(res, ret, isReverse, idsToIdx)
}

// retryBatchAssociations tries BatchAssociations up to three times, retrying if err is a connection pool exhausted
func retryBatchAssociations(ctx context.Context, ent cohesion.Entity, target string, ids []string, cohesion CohesionClient) (*cohesion.ListAssocResponseWithMeta, error) {
	i := 0
	for {
		ret, err := cohesion.BatchAssociations(ctx, ent, target, ids)
		if err == nil {
			return ret, nil
		}
		if !strings.Contains(err.Error(), "connection pool exhausted") {
			return nil, err
		}
		i++
		if i >= 3 {
			return nil, err
		}
		time.Sleep(time.Millisecond * 10)
	}
}

func (u *ActorRelationshipTraitLoader) friendsMulti(ctx context.Context, userFromID string, toActors []entity.Entity) ([]bool, error) {
	ret := make([]bool, len(toActors))
	idsToIdx := make(map[string]int, len(toActors))
	for idx, userID := range toActors {
		idsToIdx[userID.ID()] = idx
	}

	allFriends, err := u.FriendshipClient.GetFriends(ctx, userFromID, friendship.GetFriendsParams{}, nil)
	if err != nil {
		return nil, errors.Wrapf(err, "unable to discover friends of %s", userFromID)
	}
	for _, friend := range allFriends.Friends {
		friendID := friend.Entity.ID
		retIdx, exists := idsToIdx[friendID]
		if !exists {
			continue
		}
		ret[retIdx] = true
	}
	return ret, nil
}

func (u *ActorRelationshipTraitLoader) earlyExit(ctx context.Context) bool {
	u.mu.Lock()
	defer u.mu.Unlock()
	return u.Rand.Float64() <= u.SkipRelationshipLoading.Get()
}

func (u *ActorRelationshipTraitLoader) durationCheck(startTime time.Time, stat string) {
	u.Stats.TimingDurationC(stat, time.Since(startTime), .1)
}

// ForMultipleFromActors is similar to ForMultipleToActors but loads follows in reverse (followed_by rather than follows).
//  The return is actorsFrom[i] => actorTo
func (u *ActorRelationshipTraitLoader) ForMultipleFromActors(ctx context.Context, fromActors []entity.Entity, toActor entity.Entity, metadata *feeddataflow.Metadata) ([]ActorRelationshipTraits, error) {
	return u.forMultipleActors(ctx, toActor, fromActors, true, metadata)
}

// ForMultipleToActors returns traits pointing from a single node to multiple other nodes.  It checks if actorFrom follows toActor (not the other way around)
func (u *ActorRelationshipTraitLoader) ForMultipleToActors(ctx context.Context, fromActor entity.Entity, toActors []entity.Entity, metadata *feeddataflow.Metadata) ([]ActorRelationshipTraits, error) {
	return u.forMultipleActors(ctx, fromActor, toActors, false, metadata)
}

func (u *ActorRelationshipTraitLoader) forMultipleActors(ctx context.Context, fromActor entity.Entity, toActors []entity.Entity, isReverse bool, metadata *feeddataflow.Metadata) ([]ActorRelationshipTraits, error) {
	ret := make([]ActorRelationshipTraits, len(toActors))
	if u.earlyExit(ctx) {
		for i := 0; i < len(ret); i++ {
			ret[i] = ActorRelationshipTraits{
				Follows: true,
				Friends: true,
			}
		}
		return ret, nil
	}
	defer u.durationCheck(time.Now(), "for_actors.total")
	eg, egCtx := errgroup.WithContext(ctx)
	eg.Go(func() error {
		defer u.durationCheck(time.Now(), "for_actors.follow")
		follows, err := u.followsMulti(egCtx, fromActor.ID(), toActors, isReverse, metadata)
		if err != nil {
			u.Stats.IncC("for_actors.follow_err", 1, .1)
			return err
		}
		for idx := range ret {
			ret[idx].Follows = follows[idx]
		}
		return nil
	})
	eg.Go(func() error {
		defer u.durationCheck(time.Now(), "for_actors.friends")
		friends, err := u.friendsMulti(egCtx, fromActor.ID(), toActors)
		if err != nil {
			u.Stats.IncC("for_actors.friends_err", 1, .1)
			return err
		}
		for idx := range ret {
			ret[idx].Friends = friends[idx]
		}
		return nil
	})
	if err := eg.Wait(); err != nil {
		return nil, err
	}
	return ret, nil
}
