package ranker

import (
	"time"

	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"golang.org/x/net/context"
)

// EntityTraitLoader loads traits for entities
type EntityTraitLoader struct {
	DuploClient  *duplo.Client
	Stats        *service_common.StatSender
	OldDataValid *distconf.Duration
}

func (r *EntityTraitLoader) durationCheck(startTime time.Time, stat string) {
	r.Stats.TimingDurationC(stat, time.Since(startTime), .1)
}

// ForEntity loads an EntityTraits object for a single entity key
func (r *EntityTraitLoader) ForEntity(ctx context.Context, ent entity.Entity, metadata *feeddataflow.Metadata) (*EntityTraits, error) {
	if trait := r.forMetadata(ent, metadata); trait != nil {
		return trait, nil
	}
	if ent.Namespace() == entity.NamespacePost {
		defer r.durationCheck(time.Now(), "entity_stats.post")
		return r.forPost(ctx, ent)
	}
	if ent.Namespace() == entity.NamespaceComment {
		defer r.durationCheck(time.Now(), "entity_stats.comment")
		return r.forComment(ctx, ent)
	}
	if ent.Namespace() == entity.NamespaceShare {
		defer r.durationCheck(time.Now(), "entity_stats.comment")
		return r.forShare(ctx, ent)
	}
	return &EntityTraits{}, nil
}

func (r *EntityTraitLoader) forMetadata(entity entity.Entity, metadata *feeddataflow.Metadata) (ret *EntityTraits) {
	defer func() {
		if ret != nil {
			r.Stats.IncC("metadata.loaded", 1, 1)
			return
		}
		r.Stats.IncC("metadata.missed", 1, 1)
	}()
	if metadata == nil {
		return nil
	}
	oldestAllowed := time.Now().Add(-r.OldDataValid.Get())

	isDeleted, validAtDelete, existsDeleted := metadata.GetIsDeleted(entity)
	if !existsDeleted || validAtDelete.Before(oldestAllowed) {
		return nil
	}
	if isDeleted {
		return &EntityTraits{
			IsDeleted: isDeleted,
		}
	}

	creationTime, validAtCreation, existsCreation := metadata.GetEntityCreationTime(entity)
	if !existsCreation || validAtCreation.Before(oldestAllowed) {
		return nil
	}
	return &EntityTraits{
		CreationTime: creationTime,
		IsDeleted:    isDeleted,
	}
}

func (r *EntityTraitLoader) forShare(ctx context.Context, ent entity.Entity) (*EntityTraits, error) {
	shareGUID := ent.ID()
	share, err := r.DuploClient.GetShare(ctx, shareGUID)
	if err != nil {
		return nil, errors.Wrap(err, "cannot fetch share from duplo")
	}
	if share == nil {
		// It's deleted.  No time.
		return &EntityTraits{IsDeleted: true}, nil
	}
	// Hours since unix nano
	return &EntityTraits{
		CreationTime: share.CreatedAt,
	}, nil
}

func (r *EntityTraitLoader) forPost(ctx context.Context, ent entity.Entity) (*EntityTraits, error) {
	postGUID := ent.ID()
	post, err := r.DuploClient.GetPost(ctx, postGUID)
	if err != nil {
		return nil, errors.Wrap(err, "cannot fetch post from duplo")
	}
	if post == nil {
		// It's deleted.  No time.
		return &EntityTraits{IsDeleted: true}, nil
	}
	// Hours since unix nano
	return &EntityTraits{
		CreationTime: post.CreatedAt,
	}, nil
}

func (r *EntityTraitLoader) forComment(ctx context.Context, ent entity.Entity) (*EntityTraits, error) {
	commentGUID := ent.ID()
	comment, err := r.DuploClient.GetComment(ctx, commentGUID)
	if err != nil {
		return nil, errors.Wrap(err, "cannot fetch comment from duplo")
	}
	if comment == nil {
		// It's deleted.  No time.
		return &EntityTraits{IsDeleted: true}, nil
	}
	// Hours since unix nano
	return &EntityTraits{
		CreationTime: comment.CreatedAt,
	}, nil
}
