package ranker

import (
	"math"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"golang.org/x/net/context"
)

// A Story is anything that can be stored in a feed
type Story struct {
	*Activity
	StoryID string
}

// StoryBatch is a group of stories in a group of feeds
type StoryBatch struct {
	Stories  []*Story
	FeedIDs  []string
	Metadata *feeddataflow.Metadata `json:"metadata,omitempty"`
}

func (sb *StoryBatch) uniqueActors() []entity.Entity {
	s := make(map[entity.Entity]struct{}, len(sb.Stories))
	for _, story := range sb.Stories {
		s[story.Actor] = struct{}{}
	}
	ret := make([]entity.Entity, 0, len(s))
	for k := range s {
		ret = append(ret, k)
	}
	return ret
}

// DeleteStoryScore is the score given to stories that should be removed from Feed if they exist
const DeleteStoryScore = math.SmallestNonzeroFloat64

// Activity is the feed object that comes into fanout service
type Activity struct {
	Entity   entity.Entity          `json:"entity"`
	Verb     verb.Verb              `json:"verb"`
	Actor    entity.Entity          `json:"actor"`
	Metadata *feeddataflow.Metadata `json:"metadata,omitempty"`
}

// ActivityBatch is a group of activity that should be ranked across feeds
type ActivityBatch struct {
	Activities []*Activity            `json:"activities"`
	FeedIDs    []string               `json:"feed_ids"`
	Metadata   *feeddataflow.Metadata `json:"metadata,omitempty"`
}

// Ranker is an item that can score activities
type Ranker interface {
	// [len(Activity)][len(FeedIDs)]
	Score(ctx context.Context, s *ActivityBatch) (RankedActivity, error)
}

type storedActivityStruct struct {
	storyID string
	score   float64
}

// RankedActivity is a stored set of already ranked feed activity
type RankedActivity struct {
	store map[Activity]map[string]*storedActivityStruct
}

func (r *RankedActivity) merge(from RankedActivity) {
	for activity, m := range from.store {
		for feedID, store := range m {
			r.storeScore(&activity, feedID, store.storyID, store.score)
		}
	}
}

func (r *RankedActivity) storeScore(a *Activity, feedID string, storyID string, score float64) {
	if r.store == nil {
		r.store = make(map[Activity]map[string]*storedActivityStruct)
	}
	if _, exists := r.store[*a]; !exists {
		r.store[*a] = make(map[string]*storedActivityStruct)
	}
	r.store[*a][feedID] = &storedActivityStruct{
		storyID: storyID,
		score:   score,
	}
}

// Rank returns a stored storyID and score for an activity in a feed.  Returns empty values if value is unknown
func (r *RankedActivity) Rank(a *Activity, feedID string) (string, float64) {
	if r == nil || r.store == nil {
		return "", 0.0
	}
	byFeed, exists := r.store[*a]
	if !exists {
		return "", 0.0
	}
	score, exists := byFeed[feedID]
	if !exists {
		return "", 0.0
	}
	return score.storyID, score.score
}
