package ranker

import (
	"time"

	"golang.org/x/net/context"
)

// RecencyTraitScorer scores stories by how recently they happened.  Can optionally force the story to be between
// friends
type RecencyTraitScorer struct {
	RequireRelationship bool
}

var _ RankWithTraits = &RecencyTraitScorer{}

// ScoreWithTraits scores a story by how recent it is
func (r *RecencyTraitScorer) ScoreWithTraits(ctx context.Context, s *StoryBatch, tp *StoryBatchTraits) (RankedActivity, error) {
	var ret RankedActivity
	for _, feedID := range s.FeedIDs {
		feedOwner := tp.FeedTraits[feedID].Owner
		for _, activity := range s.Stories {
			storyAuthor := activity.Actor
			entityTraits := tp.EntityTraits[activity.Entity]
			if entityTraits.IsDeleted {
				ret.storeScore(activity.Activity, feedID, activity.StoryID, DeleteStoryScore)
				continue
			}
			if feedOwner.ID() != storyAuthor.ID() {
				relationshipTraits := tp.ActorRelationshipTraits[feedOwner][storyAuthor]
				if r.RequireRelationship && (!relationshipTraits.Follows && !relationshipTraits.Friends) {
					ret.storeScore(activity.Activity, feedID, activity.StoryID, DeleteStoryScore)
					continue
				}
			}
			createdAtAsFloat := r.scoreForTime(entityTraits.CreationTime)
			ret.storeScore(activity.Activity, feedID, activity.StoryID, createdAtAsFloat)
		}
	}
	return ret, nil
}

func (r *RecencyTraitScorer) scoreForTime(t time.Time) float64 {
	return t.Sub(time.Unix(0, 0)).Hours()
}
