package ranker

import (
	"sync"
	"time"

	"code.justin.tv/feeds/feeds-common/entity"
)

// EntityTraits describe feed entities
type EntityTraits struct {
	CreationTime time.Time
	IsDeleted    bool
}

// ActorTraits describe actors on a Feed
type ActorTraits struct {
}

// FeedType is the class of a feed
type FeedType int

const (
	// Unknown means we don't know the feed type
	Unknown FeedType = iota
	// Channel is the feed on the channel page
	Channel
	// News is the aggregated news feed
	News
)

// FeedTraits describe a single feed
type FeedTraits struct {
	Owner    entity.Entity
	FeedType FeedType
}

// ActorRelationshipTraits is static per ActorA -> ActorB relationship.  Is not symmetric.
// It reads as "ActorA" <XYZ> "ActorB". For example, "ActorA" "Follows" "ActorB"
type ActorRelationshipTraits struct {
	Follows bool
	Friends bool
}

// StoryBatchTraits is a loaded group of traits about a story batch
type StoryBatchTraits struct {
	EntityTraits            map[entity.Entity]*EntityTraits
	ActorTraits             map[entity.Entity]*ActorTraits
	FeedTraits              map[string]*FeedTraits
	ActorRelationshipTraits map[entity.Entity]map[entity.Entity]*ActorRelationshipTraits

	mu sync.Mutex
}

func (t *StoryBatchTraits) setEntity(ent entity.Entity, et *EntityTraits) {
	t.mu.Lock()
	defer t.mu.Unlock()
	if t.EntityTraits == nil {
		t.EntityTraits = make(map[entity.Entity]*EntityTraits)
	}
	t.EntityTraits[ent] = et
}

func (t *StoryBatchTraits) setActor(actor entity.Entity, at *ActorTraits) {
	t.mu.Lock()
	defer t.mu.Unlock()
	if t.ActorTraits == nil {
		t.ActorTraits = make(map[entity.Entity]*ActorTraits)
	}
	t.ActorTraits[actor] = at
}

func (t *StoryBatchTraits) setFeedTraits(feedID string, ft *FeedTraits) {
	t.mu.Lock()
	defer t.mu.Unlock()
	if t.FeedTraits == nil {
		t.FeedTraits = make(map[string]*FeedTraits)
	}
	t.FeedTraits[feedID] = ft
}

func (t *StoryBatchTraits) addActorRelationshipTraits(from entity.Entity, to entity.Entity, traits *ActorRelationshipTraits) {
	t.mu.Lock()
	defer t.mu.Unlock()
	if t.ActorRelationshipTraits == nil {
		t.ActorRelationshipTraits = make(map[entity.Entity]map[entity.Entity]*ActorRelationshipTraits)
	}
	if t.ActorRelationshipTraits[from] == nil {
		t.ActorRelationshipTraits[from] = make(map[entity.Entity]*ActorRelationshipTraits)
	}
	t.ActorRelationshipTraits[from][to] = traits
}
