package ranker

import "golang.org/x/net/context"

// TraitLoader can loads traits about a batch of stories for later ranking
type TraitLoader interface {
	LoadTraits(ctx context.Context, s *StoryBatch) (*StoryBatchTraits, error)
}

// StoryLoader creates a story (with a storyID) from activity for a feed
type StoryLoader interface {
	LoadStory(ctx context.Context, a *Activity) *Story
}

// RankWithTraits can rank stories with a pre loaded StoryBatchTraits
type RankWithTraits interface {
	ScoreWithTraits(ctx context.Context, s *StoryBatch, tp *StoryBatchTraits) (RankedActivity, error)
}

// TraitRanker can rank items with with a RankWithTraits
type TraitRanker struct {
	StoryLoader    StoryLoader
	TraitLoader    TraitLoader
	RankWithTraits RankWithTraits
}

var _ Ranker = &TraitRanker{}

// Score activities with a StoryBatchTraits pre loaded
func (t *TraitRanker) Score(ctx context.Context, ab *ActivityBatch) (RankedActivity, error) {
	sb := StoryBatch{
		FeedIDs:  ab.FeedIDs,
		Metadata: ab.Metadata,
	}
	for _, a := range ab.Activities {
		sb.Stories = append(sb.Stories, t.StoryLoader.LoadStory(ctx, a))
	}
	batchTraits, err := t.TraitLoader.LoadTraits(ctx, &sb)
	if err != nil {
		return RankedActivity{}, err
	}
	return t.RankWithTraits.ScoreWithTraits(ctx, &sb, batchTraits)
}
