package recommendations

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"time"

	"code.justin.tv/discovery/recommendations/client"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common/feedcache"
	"code.justin.tv/foundation/twitchclient"
	"github.com/cep21/circuit"
	"github.com/satori/go.uuid"
	"golang.org/x/sync/errgroup"
)

type recommendationsParams struct {
	DeviceID  string
	RequestID string
	Language  string
}

// RecCursor contains information used to page through a recommended feed stored in redis
type recCursor struct {
	Offset    int       `json:"o"`
	FetchTime time.Time `json:"ft"`
	BatchID   string    `json:"bid,omitempty"`
}

func (c *recCursor) encode() (string, error) {
	b, err := json.Marshal(c)
	if err != nil {
		return "", errors.Wrap(err, "cursor encode error")
	}
	encoded := base64.StdEncoding.EncodeToString(b)
	return encoded, nil
}

// DecodeRecCursor converts a cursor string to a RecCursor struct
func decodeCursor(cursor string) (*recCursor, error) {
	if cursor == "" {
		return nil, nil
	}

	b, err := base64.StdEncoding.DecodeString(cursor)
	if err != nil {
		return nil, errors.Wrap(err, "cursor decode error")
	}

	c := recCursor{}
	if err = json.NewDecoder(bytes.NewReader(b)).Decode(&c); err != nil {
		return nil, errors.Wrap(err, "cursor decode error")
	}

	return &c, nil
}

// RecommendationLoaderConfig configures RecommendationLoader
type RecommendationLoaderConfig struct {
	RecsCacheTTL         *distconf.Duration
	LoadMoreRecsAt       *distconf.Int
	RecFeedPostFrequency *distconf.Int
	RecPostStartIndex    *distconf.Int
}

// Load config for RecommendationLoader from distconf
func (r *RecommendationLoaderConfig) Load(d *distconf.Distconf) error {
	r.RecsCacheTTL = d.Duration("masonry.recs-cache-ttl", time.Hour*48)
	r.LoadMoreRecsAt = d.Int("masonry.recs-load-more-at", 30)
	r.RecFeedPostFrequency = d.Int("masonry.rec_feed_post_frequency", 5)
	r.RecPostStartIndex = d.Int("masonry.rec_post_start_index", 1)

	return nil
}

// RecommendationLoader loads recommended stories from the recs external service
type RecommendationLoader struct {
	Cache       *feedcache.ObjectCache
	Config      *RecommendationLoaderConfig
	Recs        FeedsRecsClient
	GetNewsfeed func(ctx context.Context, userID string, limit int) ([]Recommendation, error)
	Log         *log.ElevatedLog
}

// GetMoreRecommendations will return recommendations from the rec service, using redis if possible
func (r *RecommendationLoader) GetMoreRecommendations(ctx context.Context, userID string, limit int64, cursor string, deviceID, language string) (*RecommendationResponse, error) {
	oldCursor, err := decodeCursor(cursor)
	if err != nil {
		r.Log.LogCtx(ctx, "err", err, "user", userID, "cursor", cursor, "cannot decode cursor")
		oldCursor = nil
	}
	batchID := r.getNextBatchID(ctx, oldCursor)
	recParams := &recommendationsParams{
		DeviceID:  deviceID,
		Language:  language,
		RequestID: batchID,
	}
	recs, err := r.loadRecsFromCache(ctx, userID, recParams)
	if err != nil {
		return nil, err
	}

	var offset int
	if oldCursor != nil && recs.FetchTime.Equal(oldCursor.FetchTime) {
		offset = oldCursor.Offset
	}

	newOffset := offset + int(limit)
	r.Log.DebugCtx(ctx, "age", time.Since(recs.FetchTime), "offset", newOffset)
	// This is here because apparently returning [2:4] on a 4-capacity [1, 2] gives back [0, 0] and not []
	// The capacity is changed when the slice is deserialized back from cache
	if newOffset > len(recs.Recommendations) {
		newOffset = len(recs.Recommendations)
	}
	returnedRecs := recs.Recommendations[offset:newOffset]
	for i := range returnedRecs {
		returnedRecs[i].Score = float64(recs.FetchTime.Unix() - int64(offset-i))
	}

	var newCursor string
	if newOffset < len(recs.Recommendations) {
		newRecCursor := &recCursor{
			Offset:    newOffset,
			FetchTime: recs.FetchTime,
			BatchID:   batchID,
		}
		newCursor, err = newRecCursor.encode()
		if err != nil {
			return nil, err
		}
	}

	// Pass in a copy of recs so that we don't manipulate the original object
	go r.updateCache(userID, *recs, newOffset, recParams)

	return &RecommendationResponse{
		Recs:    returnedRecs,
		Cursor:  newCursor,
		BatchID: batchID,
	}, nil
}

func (r *RecommendationLoader) updateCache(userID string, recs cachedObject, newOffset int, recParams *recommendationsParams) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*60)
	defer cancel()
	itemsLeft := len(recs.Recommendations) - newOffset
	// If we are about to run out, force a cache refresh
	if r.Config.LoadMoreRecsAt.Get() >= 0 && itemsLeft <= int(r.Config.LoadMoreRecsAt.Get()) {
		if err := r.FreshenCache(ctx, userID, recParams); err != nil {
			r.Log.LogCtx(ctx, "err", err, "unable to freshen a cache")
		}
	} else if newOffset > recs.NumServed {
		recs.NumServed = newOffset
		if err := r.Cache.ForceCached(ctx, recCacheKeyForUser(userID), &recs); err != nil {
			r.Log.LogCtx(ctx, "err", err, "unable to re cache key")
		}
	}
}

// FreshenCache forces fresh recommendations into the cache for a user
func (r *RecommendationLoader) FreshenCache(ctx context.Context, userID string, recParams *recommendationsParams) error {
	newRecs, err := r.loadFreshRecs(ctx, userID, recParams)
	if err != nil {
		return err
	}
	if len(newRecs.Recommendations) <= int(r.Config.LoadMoreRecsAt.Get()) {
		return errors.Errorf("unable to load enough fresh recs: %d vs %d", len(newRecs.Recommendations), r.Config.LoadMoreRecsAt.Get())
	}
	return r.Cache.ForceCached(ctx, recCacheKeyForUser(userID), newRecs)
}

func recCacheKeyForUser(userID string) string {
	return "recs:" + userID
}

func (r *RecommendationLoader) loadRecsFromCache(ctx context.Context, userID string, recParams *recommendationsParams) (*cachedObject, error) {
	var storeInto *cachedObject
	err := r.Cache.Cached(ctx, recCacheKeyForUser(userID), func() (interface{}, error) {
		return r.loadFreshRecs(ctx, userID, recParams)
	}, &storeInto)
	if err != nil {
		return nil, err
	}
	storeInto.ttl = r.Config.RecsCacheTTL.Get()
	return storeInto, nil
}

// loadFreshRecs forces new recs, not using the cache, filters them, and returns a usable object
func (r *RecommendationLoader) loadFreshRecs(ctx context.Context, userID string, recParams *recommendationsParams) (*cachedObject, error) {
	res, err := r.fetchResult(ctx, userID, recParams)
	if err != nil {
		return nil, err
	}
	return &cachedObject{
		RecGenerationID: res.RecGenerationID,
		Recommendations: res.Recommendations,
		FetchTime:       time.Now(),
		ttl:             r.Config.RecsCacheTTL.Get(),
	}, nil
}

// fetchResult goes directly to the recs feed and converts their models to local struct types
func (r *RecommendationLoader) fetchResult(ctx context.Context, userID string, recParams *recommendationsParams) (*recResult, error) {
	params := &recommendations.GetRecommendationsParams{
		UserID:  userID,
		Context: convertToParamsContext(recParams),
	}
	var recGenerationID string
	mRecs := []Recommendation{}
	posts := []Recommendation{}
	g, gCtx := errgroup.WithContext(ctx)
	g.Go(func() error {
		dbPosts, err := r.GetNewsfeed(gCtx, userID, 20)
		if err != nil {
			r.Log.LogCtx(ctx, "err", err, "unable to fetch newsfeed")
			return err
		}
		posts = dbPosts
		return nil
	})
	g.Go(func() error {
		res, err := r.Recs.GetRecommendations(gCtx, params, nil)
		if err != nil {
			return errors.Wrap(err, "unable to fetch recommendations")
		}
		recGenerationID = res.GenerationID
		mRecs = r.convertRecsToMasonryRecs(res.Recommendations, recGenerationID)
		return nil
	})
	if err := g.Wait(); err != nil {
		return nil, err
	}
	recommendations := r.mergePostsAndRecs(posts, mRecs)
	ret := &recResult{
		RecGenerationID: recGenerationID,
		RecRequestID:    recParams.RequestID,
		Recommendations: recommendations,
	}
	return ret, nil
}

// convertRecsToMasonryRecs converts Recommendation client recs to Masonry recs
func (r *RecommendationLoader) convertRecsToMasonryRecs(recs []*recommendations.Recommendation, recGenID string) []Recommendation {
	mRecs := make([]Recommendation, len(recs))
	for i, rec := range recs {
		index := i
		mRecs[i] = Recommendation{
			Type: rec.Kind,
			ID:   rec.ID,
			Reason: RecommendationReason{
				Kind:      rec.Reason.Kind,
				ContextID: rec.Reason.ContextID,
			},
			RecGenerationID:    recGenID,
			RecGenerationIndex: &index,
		}
	}
	return mRecs
}

// mergePostsAndRecs combine recommended content and newsfeed posts into a single array
func (r *RecommendationLoader) mergePostsAndRecs(posts []Recommendation, recs []Recommendation) []Recommendation {
	recommendations := make([]Recommendation, 0, len(recs)+len(posts))
	postStartIndex := int(r.Config.RecPostStartIndex.Get())
	postFrequency := int(r.Config.RecFeedPostFrequency.Get())
	toTakeFromRecs := min(postStartIndex, len(recs))
	recommendations, recs = append(recommendations, recs[0:toTakeFromRecs]...), recs[toTakeFromRecs:]
	for len(recs) > 0 || len(posts) > 0 {
		toTakeFromPosts := min(1, len(posts))
		recommendations, posts = append(recommendations, posts[0:toTakeFromPosts]...), posts[toTakeFromPosts:]
		toTakeFromRecs = min(postFrequency, len(recs))
		recommendations, recs = append(recommendations, recs[0:toTakeFromRecs]...), recs[toTakeFromRecs:]
	}
	return recommendations
}

func (r *RecommendationLoader) getEncodedRecCursor(offset int, fetchTime time.Time, batchID string) (string, error) {
	recCursor := &recCursor{
		Offset:    offset,
		FetchTime: fetchTime,
		BatchID:   batchID,
	}

	return recCursor.encode()
}

// getNextBatchID returns the batch ID that is associated with the current run of getFeed.  The batch ID is a tracking
// parameter meant to group events associated with a user's "session" with a feed.  A fresh reload of the feed gets
// a new batch ID, while paging deeper into the feed will re-use the existing batch ID.
func (r *RecommendationLoader) getNextBatchID(ctx context.Context, oldCursor *recCursor) string {

	// When there is no batch ID, we assume that this is a fresh load of the feed, and generate a new batch ID.
	if oldCursor == nil || oldCursor.BatchID == "" {
		return uuid.NewV4().String()
	}

	return oldCursor.BatchID
}

// min returns the lower of two numbers
func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}

// convertToParamsContext converts recommendationParams into a map that has same shape as the
// RecommendationsContext struct. https://git-aws.internal.justin.tv/discovery/recommendations/blob/master/logic/models.go#L5
// The Recommendations service deserializes the body of our request into a RecommendationsContext struct.
func convertToParamsContext(recParams *recommendationsParams) map[string]interface{} {
	paramsContext := make(map[string]interface{}, 3)
	if recParams != nil {
		if recParams.DeviceID != "" {
			paramsContext["device_id"] = recParams.DeviceID
		}
		if recParams.RequestID != "" {
			paramsContext["rec_request_id"] = recParams.RequestID
		}
		if recParams.Language != "" {
			paramsContext["language"] = []string{recParams.Language}
		}
	}
	return paramsContext
}

// FeedsRecsClient wraps a recommendations client with a circuit breaker to help masonry
// better handle when the recommendations service has high latency.
type FeedsRecsClient struct {
	Client  recommendations.Client
	Config  *FeedsRecsClientConfig
	Circuit *circuit.Circuit
}

// GetRecommendations fetches a batch of recommendations from the recommendations service.
func (c *FeedsRecsClient) GetRecommendations(
	ctx context.Context,
	params *recommendations.GetRecommendationsParams,
	reqOpts *twitchclient.ReqOpts) (*recommendations.Recommendations, error) {

	var res *recommendations.Recommendations
	err := c.Circuit.Run(ctx, func(ctx context.Context) error {
		var innerErr error
		res, innerErr = c.Client.GetRecommendations(ctx, params, nil)
		return innerErr
	})

	if err != nil {
		return nil, errors.Wrap(err, "unable to fetch recommendations")
	}
	return res, nil
}

// FeedsRecsClientConfig contains configuration information for FeedsRecsClient.
type FeedsRecsClientConfig struct {
	Timeout                  *distconf.Duration
	CircuitBreakerThreshold  *distconf.Float
	CircuitBreakerMinSamples *distconf.Int
}

// Load config for FeedsRecsClientConfig from distconf
func (r *FeedsRecsClientConfig) Load(d *distconf.Distconf) error {
	r.Timeout = d.Duration("masonry.recs_client.timeout", time.Millisecond*700)
	r.CircuitBreakerThreshold = d.Float("masonry.recs_client.circuitbreaker_threshold", 0.5)
	r.CircuitBreakerMinSamples = d.Int("masonry.recs_client.circuitbreaker_min_samples", 100)
	return nil
}
