package storage

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/models"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	uuid "github.com/satori/go.uuid"
)

// StorageConfig configures the storage engine
type FeedStorageConfig struct {
	TableName      *distconf.Str
	cleanUpFollows *distconf.Bool
}

// Load storage config from distconf
func (c *FeedStorageConfig) Verify(dconf *distconf.Distconf, prefix string) error {
	configString := "masonry." + prefix + "_feed_table"
	c.TableName = dconf.Str(configString, "")
	if c.TableName.Get() == "" {
		return errors.New("unable to find storage table " + configString)
	}
	c.cleanUpFollows = dconf.Bool("masonry."+prefix+"_clean_up_follows", true)
	return nil
}

// StoryFeedStorage stores scored stories into DynamoDB
type FeedStorage struct {
	Config                 *FeedStorageConfig
	Dynamo                 *dynamodb.DynamoDB
	Stats                  *service_common.StatSender
	Log                    *log.ElevatedLog
	RequireConsistentReads bool
}

func timeToAttributeValue(t time.Time) *dynamodb.AttributeValue {
	nano := strconv.FormatInt(t.UTC().UnixNano(), 10)
	return &dynamodb.AttributeValue{N: aws.String(nano)}
}

func nowMicros() *time.Time {
	t := time.Now().UTC().Truncate(time.Microsecond)
	return &t
}

func formatFloat(f float64) string {
	return strconv.FormatFloat(f, 'f', -1, 64)
}

func attributeValueMapToStory(item map[string]*dynamodb.AttributeValue) (*models.Story, error) {
	var s map[string]string
	var f map[string]float64
	var err error
	if s, err = feedsdynamo.AwsStrings(item, []string{"feed_id", "story_id", "actor", "verb", "entity"}); err != nil {
		return nil, err
	}
	if f, err = feedsdynamo.AwsFloats(item, []string{"score"}); err != nil {
		return nil, err
	}
	actor, err := entity.Decode(s["actor"])
	if err != nil {
		return nil, err
	}
	vrb, err := verb.FromString(s["verb"])
	if err != nil {
		return nil, err
	}
	ent, err := entity.Decode(s["entity"])
	if err != nil {
		return nil, err
	}

	story := &models.Story{
		FeedID:  s["feed_id"],
		StoryID: s["story_id"],
		Activity: models.Activity{
			Actor:           actor,
			Verb:            vrb,
			Entity:          ent,
			RelevanceReason: reasonFromDynamo(item["relevance_reason"]),
		},
		Score: f["score"],
	}
	return story, nil
}

// FeedCursor contains the info necessary to page through a feed stored in DynamoDB.
type feedCursor struct {
	StoryID string  `json:"i"`
	Score   float64 `json:"s"`
	BatchID string  `json:"bid,omitempty"`
}

func decodeCursor(cursor string) (*feedCursor, error) {
	if cursor == "" {
		return nil, nil
	}

	b, err := base64.StdEncoding.DecodeString(cursor)
	if err != nil {
		return nil, errors.Wrap(err, "cursor decode error")
	}

	c := feedCursor{}
	err = json.NewDecoder(bytes.NewReader(b)).Decode(&c)
	if err != nil {
		return nil, errors.Wrap(err, "cursor decode error")
	}

	return &c, nil
}

func (c *feedCursor) encode() (string, error) {
	if c.StoryID == "" {
		return "", nil
	}

	b, err := json.Marshal(c)
	if err != nil {
		return "", errors.Wrap(err, "cursor encode error")
	}
	encoded := base64.StdEncoding.EncodeToString(b)
	return encoded, nil
}

func toExclusiveStartKey(feedID string, cursor *feedCursor) map[string]*dynamodb.AttributeValue {
	if cursor == nil || cursor.StoryID == "" {
		return nil
	}

	return map[string]*dynamodb.AttributeValue{
		"feed_id":  {S: &feedID},
		"story_id": {S: &cursor.StoryID},
		"score":    {N: aws.String(formatFloat(cursor.Score))},
	}
}

func itemsFromDynamo(dynamoItems []map[string]*dynamodb.AttributeValue) ([]*models.Activity, error) {
	stories, err := storiesFromDynamo(dynamoItems)
	if err != nil {
		return nil, err
	}
	items := make([]*models.Activity, 0, len(dynamoItems))
	for _, item := range stories {
		items = append(items, &item.Activity)
	}
	return items, nil
}

func storiesFromDynamo(dynamoItems []map[string]*dynamodb.AttributeValue) ([]*models.Story, error) {
	items := make([]*models.Story, 0, len(dynamoItems))
	for _, item := range dynamoItems {
		story, e := attributeValueMapToStory(item)
		if e != nil {
			return nil, e
		}
		items = append(items, story)
	}
	return items, nil
}

func (s *FeedStorage) storyCleanup(ctx context.Context, items []map[string]*dynamodb.AttributeValue) {
	if s.Config.cleanUpFollows.Get() {
		// I am purposely doing this in the foreground so I can x-ray, ctxlog, and timing trace these clean up operations.
		stories, err := storiesFromDynamo(items)
		if err == nil {
			removedCount := int64(0)
			for _, story := range stories {
				if story != nil && story.Activity.Entity.Namespace() == entity.NamespaceFollow {
					removedCount++
					if err := s.RemoveStory(ctx, story.FeedID, story.StoryID); err != nil {
						s.Log.LogCtx(ctx, "err", err, "feed", story.FeedID, "story", story.StoryID, "cannot remove story")
					}
				}
			}
			s.Stats.IncC("GetFeed.cleanup.removed", removedCount, 1.0)
		}
	}
}

// GetFeed returns the feed for a specific feed ID
func (s *FeedStorage) GetFeed(ctx context.Context, feedID string, limit int64, cursor string) (*models.Feed, error) {
	oldCursor, err := decodeCursor(cursor)
	if err != nil {
		s.Log.LogCtx(ctx, "err", err, "feed", feedID, "cursor", cursor, "cannot decode cursor")
		oldCursor = nil
	}
	startKey := toExclusiveStartKey(feedID, oldCursor)
	params := &dynamodb.QueryInput{
		TableName:                 aws.String(s.Config.TableName.Get()),
		IndexName:                 aws.String("feed_id-score-index"),
		KeyConditionExpression:    aws.String("feed_id = :feedid"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{":feedid": {S: &feedID}},
		ExclusiveStartKey:         startKey,
		Limit:                     aws.Int64(limit),
		ProjectionExpression:      s.getFeedProjectExpression(feedID),
		ScanIndexForward:          aws.Bool(false),
	}
	req, output := s.Dynamo.QueryRequest(params)

	err = service_common.ContextSend(ctx, req, s.Log)
	if err != nil {
		if req.IsErrorThrottle() {
			return nil, variedErr(err, "throttled request")
		}
		return nil, variedErr(err, "dynamo failure")
	}

	items, err := itemsFromDynamo(output.Items)
	if err != nil {
		return nil, err
	}

	// I am purposely doing this in the foreground so I can x-ray, ctxlog, and timing trace these clean up operations.
	s.storyCleanup(ctx, output.Items)

	nextBatchID := s.getNextBatchID(oldCursor)
	nextCursor, err := s.getNextFeedCursor(output.LastEvaluatedKey, nextBatchID)
	if err != nil {
		s.Stats.IncC("GetFeed.EncodeCursor.err", 1, 1.0)
		return nil, err
	}

	return &models.Feed{
		ID:     feedID,
		Cursor: nextCursor,
		Items:  items,
		Tracking: &models.FeedTracking{
			BatchID: nextBatchID,
		},
	}, nil
}

func (s *FeedStorage) getNextFeedCursor(lastEvaluatedKey map[string]*dynamodb.AttributeValue, nextBatchID string) (string, error) {
	// Return an empty cursor if there are no more results to page through.
	if lastEvaluatedKey == nil {
		return "", nil
	}

	storyIDMap, err := feedsdynamo.AwsStrings(lastEvaluatedKey, []string{"story_id"})
	if err != nil {
		return "", err
	}
	storyID := storyIDMap["story_id"]

	scoreMap, err := feedsdynamo.AwsFloats(lastEvaluatedKey, []string{"score"})
	if err != nil {
		return "", err
	}
	score := scoreMap["score"]

	fc := feedCursor{
		StoryID: storyID,
		Score:   score,
		BatchID: nextBatchID,
	}
	return fc.encode()
}

func (s *FeedStorage) getNextBatchID(oldCursor *feedCursor) string {
	// When there is no batch ID, we assume that this is a fresh load of the feed, and generate a new batch ID.
	if oldCursor == nil || oldCursor.BatchID == "" {
		return uuid.NewV4().String()
	}

	return oldCursor.BatchID
}

func (s *FeedStorage) getFeedProjectExpression(feedID string) *string {
	projectionExp := "feed_id,story_id,actor,verb,entity,score"

	// The relevance_reason & rec_generation_id attributes are not a part of the feed_id-score-index index, and fetching them
	// results in an extra read from the base table.  To avoid this extra read, we only include relevance_reason & rec_generation_id
	// when getting a recommended feed, since it is the only feed type that uses reasons & rec_generation_id.
	// TODO: Remove this special casing when we have an index that includes relevance_reason and rec_generation_id.
	if strings.HasPrefix(feedID, "r:") {
		projectionExp += ",relevance_reason,rec_generation_id"
	}

	return &projectionExp
}

// errVariedErrString is a type of error where the Error() string contains random junk.  We cannot rollbar group
// by this random junk, so instead group by the stack trace only
type errVariedErrString struct {
	error
	msg string
}

// Makes it easier to group error messages for rollbar
func (e *errVariedErrString) Fingerprint() string {
	return e.msg
}

func variedErr(err error, msg string) error {
	return errors.Wrap(&errVariedErrString{
		error: err,
		msg:   msg,
	}, msg)
}

// SaveStory puts a scored item into dynamodb
func (s *FeedStorage) SaveStory(ctx context.Context, feedID, storyID string, activity models.Activity, score float64, expiresAt *time.Time) (*models.Story, error) {
	updatedAt := nowMicros()

	story := &models.Story{
		FeedID:    feedID,
		StoryID:   storyID,
		Score:     score,
		Activity:  activity,
		UpdatedAt: *updatedAt,
	}

	items := map[string]*dynamodb.AttributeValue{
		"feed_id":    {S: &story.FeedID},
		"story_id":   {S: &story.StoryID},
		"actor":      {S: aws.String(story.Activity.Actor.Encode())},
		"verb":       {S: aws.String(string(story.Activity.Verb))},
		"entity":     {S: aws.String(story.Activity.Entity.Encode())},
		"score":      {N: aws.String(formatFloat(story.Score))},
		"updated_at": timeToAttributeValue(story.UpdatedAt),
	}

	reasonAttributeVal := reasonToDynamo(activity.RelevanceReason)
	if reasonAttributeVal != nil {
		items["relevance_reason"] = reasonAttributeVal
	}
	if activity.RecGenerationID != "" {
		items["rec_generation_id"] = &dynamodb.AttributeValue{S: aws.String(activity.RecGenerationID)}
	}
	if expiresAt != nil {
		items["expires_at"] = &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(expiresAt.Unix(), 10))}
	}

	params := &dynamodb.PutItemInput{
		TableName: aws.String(s.Config.TableName.Get()),
		Item:      items,
	}

	req, _ := s.Dynamo.PutItemRequest(params)
	startTime := time.Now()
	if err := service_common.ContextSend(ctx, req, s.Log); err != nil {
		if req.IsErrorThrottle() {
			s.Stats.IncC("SaveStory.Dynamo.PutItem.throttle", 1, 1.0)
			return nil, variedErr(err, "throttled request")
		}
		s.Stats.IncC("SaveStory.Dynamo.PutItem.err", 1, 1.0)
		return nil, variedErr(err, "dynamodb putitem error")
	}
	s.Stats.TimingDurationC("SaveStory.Dynamo.PutItem", time.Since(startTime), 1.0)
	return story, nil
}

// RemoveStory removes a story from dynamodb
func (s *FeedStorage) RemoveStory(ctx context.Context, feedID, storyID string) error {
	params := &dynamodb.DeleteItemInput{
		TableName: aws.String(s.Config.TableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"feed_id":  {S: &feedID},
			"story_id": {S: &storyID},
		},
	}

	req, _ := s.Dynamo.DeleteItemRequest(params)
	startTime := time.Now()
	if err := service_common.ContextSend(ctx, req, s.Log); err != nil {
		if req.IsErrorThrottle() {
			s.Stats.IncC("RemoveStory.Dynamo.DeleteItem.throttle", 1, 1.0)
			return variedErr(err, "throttled request")
		}
		s.Stats.IncC("RemoveStory.Dynamo.DeleteItem.err", 1, 1.0)
		return variedErr(err, "dynamodb DeleteItem error")
	}
	s.Stats.TimingDurationC("RemoveStory.Dynamo.DeleteItem", time.Since(startTime), 1.0)
	return nil
}

func reasonToDynamo(reason *models.RelevanceReason) *dynamodb.AttributeValue {
	if reason == nil {
		return nil
	}

	keyToAttributeVal := make(map[string]*dynamodb.AttributeValue, 3)
	if reason.Kind != "" {
		keyToAttributeVal["kind"] = &dynamodb.AttributeValue{S: &reason.Kind}
	}
	if reason.EntityID != "" {
		keyToAttributeVal["entity_id"] = &dynamodb.AttributeValue{S: &reason.EntityID}
	}
	if reason.Source != "" {
		keyToAttributeVal["source"] = &dynamodb.AttributeValue{S: &reason.Source}
	}

	if len(keyToAttributeVal) == 0 {
		return nil
	}
	return &dynamodb.AttributeValue{
		M: keyToAttributeVal,
	}
}

func reasonFromDynamo(val *dynamodb.AttributeValue) *models.RelevanceReason {
	if val == nil {
		return nil
	}

	mapAttr := val.M
	if mapAttr == nil {
		return nil
	}

	var kind string
	if kindAttribute, ok := mapAttr["kind"]; ok && kindAttribute.S != nil {
		kind = *kindAttribute.S
	}

	var entityID string
	if entityIDAttribute, ok := mapAttr["entity_id"]; ok && entityIDAttribute.S != nil {
		entityID = *entityIDAttribute.S
	}

	var source string
	if sourceAttribute, ok := mapAttr["source"]; ok && sourceAttribute.S != nil {
		source = *sourceAttribute.S
	}

	return &models.RelevanceReason{
		Kind:     kind,
		EntityID: entityID,
		Source:   source,
	}
}
