package main

import (
	"math/rand"
	"net"
	"net/http"
	"os"
	"time"

	"os/signal"

	"expvar"

	"code.justin.tv/chat/friendship/client"
	rec_client "code.justin.tv/discovery/recommendations/client"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/masonry"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/providers"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/ranker"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/recommendations"
	"code.justin.tv/feeds/masonry/cmd/masonry/internal/storage"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdim"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedcache/gorediscache"
	"code.justin.tv/feeds/service-common/feedsqs"
	xraydynamo "code.justin.tv/feeds/xray/plugins/dynamodb"
	"code.justin.tv/feeds/xray/plugins/ec2"
	"code.justin.tv/foundation/twitchclient"
	cohesion "code.justin.tv/web/cohesion/client/v2"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/cep21/circuit"
	"github.com/go-redis/redis"
)

const (
	teamName    = "feeds"
	serviceName = "masonry"
)

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{
		CounterMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "redis.cache.*",
				DimensionsMap: "cache_type.%.%",
			},
			{
				MetricPath:    "redis.close_err",
				DimensionsMap: "cache_type.%",
				MetricName:    "cache_error",
			},
			{
				MetricPath:    "storage.*.*.Dynamo.*.*",
				DimensionsMap: "%.feed_type.%.method.%.dynamo_method.*",
				MetricName:    "error_count",
			},
			{
				MetricPath:    "storage.*.*.*.*",
				DimensionsMap: "%.feed_type.method.%.%",
			},
			{
				MetricPath:    "http.*.code.*",
				DimensionsMap: "%.method.%.http_code",
			},
			{
				MetricPath:    "trait_loader.**.load_traits_err",
				DimensionsMap: "-.loader_type.%",
				MetricName:    "load_error",
			},
			{
				MetricPath:    "trait_loader.*.metadata.*",
				DimensionsMap: "%.loader_type.%.is_cached",
			},
			{
				MetricPath:    "sqs_source.*.msg_input",
				DimensionsMap: "%.queue_priority.%",
			},
		},
		TimingMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "storage.*.*.Dynamo.*",
				DimensionsMap: "%.feed_type.method.%.dynamo_function",
				MetricName:    "timing",
			},
			{
				MetricPath:    "http.*.time",
				DimensionsMap: "%.method.%",
			},
			{
				MetricPath:    "sqs_source.*.sqs.*",
				DimensionsMap: "%.queue_priority.%.method",
				MetricName:    "method_timing",
			},
			{
				MetricPath:    "sqs_source.*.*.time",
				DimensionsMap: "%.queue_priority.method.%",
				MetricName:    "metric_times",
			},
			{
				MetricPath:    "trait_loader.*.load_traits",
				DimensionsMap: "%.loader_type.%",
				MetricName:    "load_timing",
			},
			{
				MetricPath:    "trait_loader.*.*.*",
				DimensionsMap: "%.loader_type.stat_type.entity_type",
				MetricName:    "trait_timing",
			},
		},
	}
}

// CodeVersion is set by build script
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CircuitConfig: map[string]circuit.Config{
			"feeds-rec": {
				Execution: circuit.ExecutionConfig{
					// This service is very slow and we sometimes have to wait up to 3 seconds
					Timeout:               time.Second * 3,
					MaxConcurrentRequests: 60,
				},
			},
		},
		CodeVersion:    CodeVersion,
		SfxSetupConfig: sfxStastdConfig(),
	},
}

type injectables struct {
	RecommendationsClient rec_client.Client
}

type service struct {
	injectables
	osExit        func(code int)
	Runner        service_common.ServiceRunner
	serviceCommon service_common.ServiceCommon

	feedMux       providers.FeedMux
	processor     masonry.BatchProcessor
	sqsNormalIn   masonry.SQSInput
	sqsPriorityIn masonry.SQSInput
	sqsHighPriIn  masonry.SQSInput
	server        masonry.HTTPServer
	duploClient   duplo.Client
	sigChan       chan os.Signal
	onListen      func(listeningAddr net.Addr)
	redisClient   *redis.ClusterClient

	cohesionFollows  cohesion.Client
	friendshipClient friendship.Client

	feedsRecsClient recommendations.FeedsRecsClient

	configs struct {
		config     config
		httpConfig masonry.HTTPConfig

		channelFeedProviderConfig         providers.ChannelFeedProviderConfig
		newsFeedProviderConfig            providers.NewsFeedProviderConfig
		recommendationsFeedProviderConfig providers.RecommendationsFeedProviderConfig

		sqsHighPriorityInputConfig masonry.SQSHighPriorityInputConfig
		sqsMidPriorityInputConfig  masonry.SQSMidPriorityInputConfig
		sqsLowPriorityInputConfig  masonry.SQSLowPriorityInputConfig

		duploConfig                duplo.Config
		redisConfig                masonry.RedisConfig
		recommendationLoaderConfig recommendations.RecommendationLoaderConfig
		feedsRecsClientConfig      recommendations.FeedsRecsClientConfig
	}
}

type config struct {
	BackoffMultiplier  *distconf.Float
	SleepBackoff       *distconf.Duration
	MaxSleepTime       *distconf.Duration
	cohesionFollowsURL *distconf.Str
	friendshipURL      *distconf.Str
	recServiceURL      *distconf.Str
	skipTraitLoading   *distconf.Float
	followSplitSize    *distconf.Int
	oldDataValid       *distconf.Duration
}

func (c *config) Load(d *distconf.Distconf) error {
	c.BackoffMultiplier = d.Float("masonry.storage_backoff.multiplier", 3.0)
	c.SleepBackoff = d.Duration("masonry.storage_backoff.sleep_backoff", time.Millisecond*100)
	c.MaxSleepTime = d.Duration("masonry.storage_backoff.max_sleep_time", time.Second*5)
	c.cohesionFollowsURL = d.Str("candidatediscovery.follows.cohesion_rpc_url", "")
	c.recServiceURL = d.Str("recommendation-service.url", "")
	c.skipTraitLoading = d.Float("masonry.skip_trait_loading", 1.0)
	c.oldDataValid = d.Duration("masonry.old_data_valid_til", time.Second*45)
	c.followSplitSize = d.Int("masonry.cohesion_split_size", 100)

	if c.cohesionFollowsURL.Get() == "" {
		return errors.New("unable to find a valid candidatediscovery.follows.cohesion_rpc_url")
	}
	if c.recServiceURL.Get() == "" {
		return errors.New("unable to find a valid recommendation-service.url")
	}
	c.friendshipURL = d.Str("candidatediscovery.friends.friendship_url", "")
	if c.friendshipURL.Get() == "" {
		return errors.New("unable to find a valid candidatediscovery.friends.friendship_url")
	}
	return nil
}

func (s *service) setup() error {
	if err := s.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(
		s.serviceCommon.Config,
		&s.configs.config,
		&s.configs.httpConfig,

		&s.configs.channelFeedProviderConfig,
		&s.configs.newsFeedProviderConfig,
		&s.configs.recommendationsFeedProviderConfig,

		&s.configs.sqsHighPriorityInputConfig,
		&s.configs.sqsMidPriorityInputConfig,
		&s.configs.sqsLowPriorityInputConfig,

		&s.configs.duploConfig,
		&s.configs.redisConfig,
		&s.configs.recommendationLoaderConfig,
		&s.configs.feedsRecsClientConfig,
	); err != nil {
		return err
	}
	var err error
	rtWrappers := []func(http.RoundTripper) http.RoundTripper{
		s.serviceCommon.XRay.RoundTripper,
	}

	// TODO: cohesion client does not support xray.  It should allow customizing the HTTP or gRPC calls
	if s.cohesionFollows, err = cohesion.New(s.configs.config.cohesionFollowsURL.Get(), "feeds-masonry"); err != nil {
		return err
	}

	if s.friendshipClient, err = friendship.NewClient(twitchclient.ClientConf{Host: s.configs.config.friendshipURL.Get(), RoundTripperWrappers: rtWrappers}); err != nil {
		return err
	}

	if s.RecommendationsClient == nil {
		recClientConf := twitchclient.ClientConf{
			Host:                 s.configs.config.recServiceURL.Get(),
			RoundTripperWrappers: rtWrappers,
			Stats:                s.serviceCommon.Statsd,
		}
		if s.RecommendationsClient, err = rec_client.NewClient(recClientConf); err != nil {
			return err
		}
	}

	s.feedsRecsClient = recommendations.FeedsRecsClient{
		Config:  &s.configs.feedsRecsClientConfig,
		Client:  s.RecommendationsClient,
		Circuit: s.serviceCommon.Circuit.MustCreateCircuit("feeds-rec"),
	}
	// TODO: Setup cohesion with redis
	if err := s.setupRedis(); err != nil {
		return err
	}

	if err := s.serviceCommon.XRay.WithPlugin(&ec2.Plugin{}); err != nil {
		s.serviceCommon.Log.Log("err", err, "unable to setup ec2 plugin")
	}

	if err := s.serviceCommon.XRay.WithPlugin(&xraydynamo.Plugin{}); err != nil {
		s.serviceCommon.Log.Log("err", err, "unable to setup dynamo plugin")
	}

	return nil
}

type goredisExpvar struct {
	client *redis.ClusterClient
}

func (g *goredisExpvar) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		return map[string]string{
			"clients":      g.client.ClientList().String(),
			"cluster_info": g.client.ClusterInfo().String(),
		}
	})
}

func (s *service) setupRedis() error {
	opts := s.configs.redisConfig.ClusterOptions()
	s.redisClient = redis.NewClusterClient(opts)
	err := s.redisClient.Ping().Err()
	if err != nil {
		return errors.Wrap(err, "unable to ping redis host")
	}
	return nil
}

func (s *service) makeSQSInput(session client.ConfigProvider, awsConf []*aws.Config, priorityLevel string, conf *feedsqs.SQSQueueProcessorConfig) masonry.SQSInput {
	sqsClient := sqs.New(session, awsConf...)
	s.serviceCommon.XRay.AWS(sqsClient.Client)
	return masonry.SQSInput{
		Destination: &s.processor,
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Log:             s.serviceCommon.Log,
			Ch:              &s.serviceCommon.Ctxlog,
			StopWaitChannel: make(chan struct{}),
			Sqs:             sqsClient,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("sqs_source." + priorityLevel),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Conf: conf,
		},
	}
}

func (s *service) inject() {
	httpClient := s.serviceCommon.XRay.Client(&http.Client{})
	redisStats := &service_common.StatSender{
		SubStatter:   s.serviceCommon.Statsd.NewSubStatter("redis"),
		ErrorTracker: &s.serviceCommon.ErrorTracker,
	}

	s.duploClient = duplo.Client{
		Config: &s.configs.duploConfig,
		RequestDoer: &ctxlog.LoggedDoer{
			Logger: s.serviceCommon.Log,
			C:      &s.serviceCommon.Ctxlog,
			Client: httpClient,
		},
		NewHTTPRequest: ctxlog.WrapHTTPRequestWithCtxlog(&s.serviceCommon.Ctxlog, http.NewRequest),
	}
	session, awsConf := service_common.CreateAWSSession(s.serviceCommon.Config)
	s.sqsHighPriIn = s.makeSQSInput(session, awsConf, "high", &s.configs.sqsHighPriorityInputConfig.SQSQueueProcessorConfig)
	s.sqsNormalIn = s.makeSQSInput(session, awsConf, "mid", &s.configs.sqsMidPriorityInputConfig.SQSQueueProcessorConfig)
	s.sqsPriorityIn = s.makeSQSInput(session, awsConf, "low", &s.configs.sqsLowPriorityInputConfig.SQSQueueProcessorConfig)

	dynamoConfig := append([]*aws.Config{
		{
			MaxRetries: aws.Int(10),
		}},
		awsConf...)

	dynamoClient := dynamodb.New(session, dynamoConfig...)
	s.serviceCommon.XRay.AWS(dynamoClient.Client)

	cacheClient := gorediscache.NewGoredisCache(s.redisClient, &s.configs.redisConfig.GoredisConfig, s.serviceCommon.Environment+":masonry:", s.serviceCommon.Log, redisStats)

	newsFeedStorage := &storage.FeedStorage{
		Config: &s.configs.newsFeedProviderConfig.FeedStorageConfig,
		Dynamo: dynamoClient,
		Stats: &service_common.StatSender{
			SubStatter:   s.serviceCommon.Statsd.NewSubStatter("storage.news_feed"),
			ErrorTracker: &s.serviceCommon.ErrorTracker,
		},
		Log: s.serviceCommon.Log,
	}
	newsFeedProvider := &providers.NewsFeedProvider{
		Config:      &s.configs.newsFeedProviderConfig,
		FeedStorage: newsFeedStorage,
		Log:         s.serviceCommon.Log,
	}

	channelFeedStorage := &storage.FeedStorage{
		Config: &s.configs.channelFeedProviderConfig.FeedStorageConfig,
		Dynamo: dynamoClient,
		Stats: &service_common.StatSender{
			SubStatter:   s.serviceCommon.Statsd.NewSubStatter("storage.channel_feed"),
			ErrorTracker: &s.serviceCommon.ErrorTracker,
		},
		Log: s.serviceCommon.Log,
	}
	channelFeedProvider := &providers.ChannelFeedProvider{
		Config:      &s.configs.channelFeedProviderConfig,
		FeedStorage: channelFeedStorage,
		Cache:       cacheClient,
		Log:         s.serviceCommon.Log,
	}

	recommendationsFeedProvider := &providers.RecommendationsFeedProvider{
		Config: &s.configs.recommendationsFeedProviderConfig,
		RecLoader: &recommendations.RecommendationLoader{
			Cache:       cacheClient,
			Config:      &s.configs.recommendationLoaderConfig,
			Recs:        s.feedsRecsClient,
			GetNewsfeed: newsFeedProvider.GetNewsfeedForRecommendations,
			Log:         s.serviceCommon.Log,
		},
		Log: s.serviceCommon.Log,
	}

	s.feedMux = providers.FeedMux{
		Log: s.serviceCommon.Log,
	}
	s.feedMux.AddProvider(providers.RecommendationsFeedMatcher, recommendationsFeedProvider)
	s.feedMux.AddMutableProvider(providers.ChannelFeedMatcher, channelFeedProvider)
	s.feedMux.AddMutableProvider(providers.NewsFeedMatcher, newsFeedProvider)

	traitLoader := ranker.SerialTraitLoader{
		ActorRelationshipTraitLoader: ranker.ActorRelationshipTraitLoader{
			Cohesion:         s.cohesionFollows,
			FriendshipClient: s.friendshipClient,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("trait_loader.actor_relationship"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Rand: rand.New(rand.NewSource(0)),
			SkipRelationshipLoading: s.configs.config.skipTraitLoading,
			MaxSplitSize:            s.configs.config.followSplitSize,
			MetadataCacheValidTill:  s.configs.config.oldDataValid,
		},
		ActorTraitLoader: ranker.ActorTraitLoader{},
		EntityTraitLoader: ranker.EntityTraitLoader{
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("trait_loader.entity"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			DuploClient:  &s.duploClient,
			OldDataValid: s.configs.config.oldDataValid,
		},
		FeedTraitLoader: ranker.FeedTraitLoader{},
		Stats: &service_common.StatSender{
			SubStatter:   s.serviceCommon.Statsd.NewSubStatter("trait_loader.serial"),
			ErrorTracker: &s.serviceCommon.ErrorTracker,
		},
	}

	var newsFeed ranker.NewsFeed
	var channelFeed ranker.ChannelFeed

	newsFeed.Ranker = &ranker.TraitRanker{
		StoryLoader: &newsFeed,
		TraitLoader: &traitLoader,
		RankWithTraits: &ranker.RecencyTraitScorer{
			RequireRelationship: true,
		},
	}

	channelFeed.Ranker = &ranker.TraitRanker{
		StoryLoader: &channelFeed,
		TraitLoader: &traitLoader,
		RankWithTraits: &ranker.RecencyTraitScorer{
			RequireRelationship: false,
		},
	}

	rankers := ranker.RankSplitter{
		RankerList: []ranker.FeedTypeRanker{
			&newsFeed, &channelFeed,
		},
	}
	s.processor = masonry.BatchProcessor{
		Ranker:  &rankers,
		Log:     s.serviceCommon.Log,
		FeedMux: &s.feedMux,
		Backoff: &service_common.ThrottledBackoff{
			Multiplier:   s.configs.config.BackoffMultiplier.Get(),
			SleepBackoff: s.configs.config.SleepBackoff.Get(),
			MaxSleepTime: s.configs.config.MaxSleepTime.Get(),
			Rand:         *rand.New(rand.NewSource(1)),
		},
	}
	s.server = masonry.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Log:         s.serviceCommon.Log,
			Config:      &s.configs.httpConfig.BaseHTTPServerConfig,
			ElevateKey:  s.serviceCommon.ElevateLogKey,
			Ctxlog:      &s.serviceCommon.Ctxlog,
			Dims:        &s.serviceCommon.CtxDimensions,
			OnListen:    s.onListen,
			PanicLogger: s.serviceCommon.PanicLogger,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			XRay: s.serviceCommon.XRay,
		},

		FeedMux:              &s.feedMux,
		PrivateBatchReciever: &s.processor,
	}
	s.server.BaseHTTPServer.SetupRoutes = s.server.SetupRoutes
	s.Runner = service_common.ServiceRunner{
		Log: s.serviceCommon.Log,
		Services: []service_common.Service{
			&s.server, &s.sqsHighPriIn, &s.sqsNormalIn, &s.sqsPriorityIn, &s.serviceCommon,
		},
		SigChan:      s.sigChan,
		SignalNotify: signal.Notify,
	}
	s.serviceCommon.ExpvarHandler.Exported["cluster"] = (&goredisExpvar{client: s.redisClient}).Var()
	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http", "github.com/rubyist/circuitbreaker"},
	}).Check(s, s.Runner)
	res.MustBeEmpty(os.Stderr)
}

func (s *service) main() {
	if err := s.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		s.osExit(1)
		return
	}
	s.inject()
	if err := s.Runner.Execute(); err != nil {
		service_common.SetupLogger.Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.serviceCommon.Log.Log("done with service")
}

func main() {
	instance.main()
}
