locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "masonry"

  account_ids = {
    integration = "724951484461"
    staging     = "724951484461"
    production  = "914569885343"
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  rec_feed_read_capacities = {
    integration = 100
    staging     = 1000
    production  = 3000
  }

  rec_feed_read_capacity = "${lookup(local.rec_feed_read_capacities, terraform.workspace)}"

  rec_feed_write_capacities = {
    integration = 400
    staging     = 4000
    production  = 5000
  }

  rec_feed_write_capacity = "${lookup(local.rec_feed_write_capacities, terraform.workspace)}"

  story_feed_read_capacities = {
    integration = 100
    staging     = 1000
    production  = 4000
  }

  story_feed_read_capacity = "${lookup(local.story_feed_read_capacities, terraform.workspace)}"

  story_feed_write_capacities = {
    integration = 400
    staging     = 4000
    production  = 30000
  }

  story_feed_write_capacity = "${lookup(local.story_feed_write_capacities, terraform.workspace)}"

  alarm_threshold_percents = {
    integration = 70
    staging     = 70
    production  = 70
  }

  alarm_threshold_percent = "${lookup(local.alarm_threshold_percents, terraform.workspace)}"

  alarm_periods = {
    integration = 60
    staging     = 60
    production  = 60
  }

  alarm_period = "${lookup(local.alarm_periods, terraform.workspace)}"

  profile = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/masonry-dynamodb-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

resource "aws_dynamodb_table" "rec_feed" {
  name           = "${local.service}_${terraform.workspace}_rec_feed"
  read_capacity  = "${local.rec_feed_read_capacity}"
  write_capacity = "${local.rec_feed_write_capacity}"
  hash_key       = "feed_id"
  range_key      = "story_id"

  attribute {
    name = "feed_id"
    type = "S"
  }

  attribute {
    name = "story_id"
    type = "S"
  }

  attribute {
    name = "score"
    type = "N"
  }

  ttl {
    attribute_name = "expires_at"
    enabled        = true
  }

  lifecycle {
    prevent_destroy = true
  }

  local_secondary_index {
    name            = "feed_id-score-index"
    range_key       = "score"
    projection_type = "ALL"
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module rec_feed_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.rec_feed.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.rec_feed.read_capacity}"
  write_capacity = "${aws_dynamodb_table.rec_feed.write_capacity}"
}

resource "aws_dynamodb_table" "story_feed" {
  name           = "${local.service}_${terraform.workspace}_story_feed"
  read_capacity  = "${local.story_feed_read_capacity}"
  write_capacity = "${local.story_feed_write_capacity}"
  hash_key       = "feed_id"
  range_key      = "story_id"

  attribute {
    name = "feed_id"
    type = "S"
  }

  attribute {
    name = "story_id"
    type = "S"
  }

  attribute {
    name = "score"
    type = "N"
  }

  ttl {
    attribute_name = "expires_at"
    enabled        = true
  }

  lifecycle {
    prevent_destroy = true
  }

  local_secondary_index {
    name               = "feed_id-score-index"
    range_key          = "score"
    projection_type    = "INCLUDE"
    non_key_attributes = ["actor", "verb", "entity"]
  }

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }
}

module story_feed_dynamodb_cloudwatch {
  source         = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//dynamodb/table_cloudwatch?ref=v0.8.20"
  table_name     = "${aws_dynamodb_table.story_feed.name}"
  alerts_sns_arn = "${data.terraform_remote_state.region_core.pagerduty_alarm_arn}"
  read_capacity  = "${aws_dynamodb_table.story_feed.read_capacity}"
  write_capacity = "${aws_dynamodb_table.story_feed.write_capacity}"
}

output story_feed_arn {
  value = "${aws_dynamodb_table.story_feed.arn}"
}

output rec_feed_arn {
  value = "${aws_dynamodb_table.rec_feed.arn}"
}
