package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/assert"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/backend/backendtest"
)

func serverWithTestBackend(t *testing.T) (*Server, *backend.Backend) {
	b := backend.NewTestBackend()

	stats, err := statsd.NewNoopClient()
	assert.Nil(t, err)

	s, err := NewServer(b, stats)
	assert.Nil(t, err)

	return s, b
}

func doTestHTTP(t *testing.T, s *Server, method, path string) *httptest.ResponseRecorder {
	url := fmt.Sprintf("%s%s", "http://localhost", path)

	r, err := http.NewRequest(method, url, nil)
	assert.Nil(t, err)

	r.Header.Set("Content-Type", "application/json")

	w := httptest.NewRecorder()
	s.ServeHTTP(w, r)

	return w
}

// Used to validate multiple calls with possible ordering of kinds (asynchUpdate/Delete/Fetch functionality)
func validateCallsWithKinds(t *testing.T, calls []backendtest.Call, expected backendtest.Call, kinds []string) {
	for _, call := range calls {
		assert.Equal(t, expected.Name, call.Name)
		assert.Equal(t, expected.E1, call.E1)
		assert.Equal(t, call.Kind, kinds)
		if expected.E2 != "" {
			assert.Equal(t, expected.E2, call.E2)
		}
		if expected.EntityKind != "" {
			assert.Equal(t, expected.EntityKind, call.EntityKind)
		}

		for k, v := range expected.Databag {
			assert.Equal(t, call.Databag[k], v)
		}
	}
}

// Used to validate multiple calls with different entity ordering (asynchCount functionality)
func validateCallsWithEntities(t *testing.T, calls []backendtest.Call, expected backendtest.Call, entities []string) {
	for _, call := range calls {
		assert.Equal(t, expected.Name, call.Name)
		assert.Equal(t, call.E1, entities)
		assert.Equal(t, expected.Kind, call.Kind)
		assert.Equal(t, expected.EntityKind, call.EntityKind)
	}
}

func validateCall(t *testing.T, call, expected backendtest.Call) {
	assert.Equal(t, expected.Name, call.Name)
	assert.Equal(t, expected.E1, call.E1)
	assert.Equal(t, expected.Kind, call.Kind)
	if expected.E2 != "" {
		assert.Equal(t, expected.E2, call.E2)
	}
	if expected.EntityKind != "" {
		assert.Equal(t, expected.EntityKind, call.EntityKind)
	}

	for k, v := range expected.Databag {
		assert.Equal(t, call.Databag[k], v)
	}
}
