package api

import (
	"fmt"
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/backend/backendtest"
	"code.justin.tv/chat/friendship/app/internal/cursor"
)

func TestGetRequestsV1(t *testing.T) {
	t.Run("should fail when the limit param is invalid", func(t *testing.T) {
		server, _ := serverWithTestBackend(t)

		path := fmt.Sprintf("/v1/users/%s/friends/requests?limit=0", backend.AliceID)

		w := doTestHTTP(t, server, "GET", path)
		assert.Equal(t, http.StatusBadRequest, w.Code)
		assert.Contains(t, w.Body.String(), http.StatusText(w.Code))
	})

	t.Run("should call the appropriate Cohesion client functions for users", func(t *testing.T) {
		server, b := serverWithTestBackend(t)
		mockFriends := b.Friends.(*backendtest.CohesionClient)

		path := fmt.Sprintf("/v1/users/%s/friends/requests", backend.AliceID)

		doTestHTTP(t, server, "GET", path)
		assert.Equal(t, 4, len(mockFriends.GetCalls()))

		expected := []backendtest.Call{
			{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByUser,
				EntityKind: backend.User,
			},
			{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			},
			{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByUser,
				EntityKind: backend.User,
			},
			{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			},
		}
		for i, call := range mockFriends.GetCalls() {
			validateCall(t, call, expected[i])
		}
	})

	t.Run("should call the appropriate Cohesion client functions for strangers", func(t *testing.T) {
		server, b := serverWithTestBackend(t)
		mockFriends := b.Friends.(*backendtest.CohesionClient)

		c := cursor.New(backend.RequestedByStranger, "foobar").Encode()
		path := fmt.Sprintf("/v1/users/%s/friends/requests?cursor=%s", backend.AliceID, c)

		doTestHTTP(t, server, "GET", path)
		assert.Equal(t, 3, len(mockFriends.GetCalls()))

		expected := []backendtest.Call{
			{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByUser,
				EntityKind: backend.User,
			},
			{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			},
			{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			},
		}
		actual := mockFriends.GetCalls()

		for i, call := range actual {
			validateCall(t, call, expected[i])
		}
	})
}

func TestGetRequestsV0(t *testing.T) {
	t.Run("When finding friend requests", func(t *testing.T) {
		t.Run("should fail when the limit param is invalid", func(*testing.T) {
			server, _ := serverWithTestBackend(t)

			path := fmt.Sprintf("/users/%s/friends/requests?limit=0", backend.AliceID)

			w := doTestHTTP(t, server, "GET", path)
			assert.Equal(t, http.StatusBadRequest, w.Code)
			assert.Contains(t, w.Body.String(), http.StatusText(w.Code))
		})

		t.Run("should call the appropriate Cohesion client functions for users", func(t *testing.T) {
			server, b := serverWithTestBackend(t)
			mockFriends := b.Friends.(*backendtest.CohesionClient)

			path := fmt.Sprintf("/users/%s/friends/requests", backend.AliceID)

			doTestHTTP(t, server, "GET", path)
			assert.Equal(t, 4, len(mockFriends.GetCalls()))

			calls := mockFriends.GetCalls()

			expected := backendtest.Call{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByUser,
				EntityKind: backend.User,
			}
			validateCall(t, calls[0], expected)

			expected = backendtest.Call{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			}
			validateCall(t, calls[1], expected)

			expected = backendtest.Call{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByUser,
				EntityKind: backend.User,
			}
			validateCall(t, calls[2], expected)

			expected = backendtest.Call{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.RequestedByStranger,
				EntityKind: backend.User,
			}
			validateCall(t, calls[3], expected)
		})
	})
}
