package api

import (
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/friendship/app/api/utils"
	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/internal"
	"code.justin.tv/chat/friendship/app/internal/cursor"
	"code.justin.tv/chat/friendship/client"
	"code.justin.tv/chat/golibs/gojiplus"
)

const defaultOutgoingRequestsLimit = 25

// getOutgoingRequestsV1 retrieves Outgoing friends requests for a user.
func (s *Server) getOutgoingRequestsV1(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	required, err := gojiplus.FetchRequiredURLParams(ctx, "user_id")
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}
	userID := required["user_id"]

	limit, err := requestLimit(r, defaultOutgoingRequestsLimit, internal.MaxOutgoingFriendRequests)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	cursorRelationship := backend.RequestToUser
	cursorToken := ""
	if c := requestCursor(r); c != "" {
		cursorObj, err := cursor.Parse(c)
		if err != nil {
			gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
			return
		}
		cursorRelationship = cursorObj.Relationship
		cursorToken = cursorObj.Cursor
	}

	sort, err := requestSort(r, friendship.DescendingDir)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	resp, err := s.backend.GetOutgoingRequestsV1(ctx, userID, cursorRelationship, cursorToken, sort, limit)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, utils.ErrorStatus(err))
		return
	}

	encodedCursor := cursor.New(resp.CursorRelationship, resp.Cursor).Encode()
	gojiplus.ServeJSON(w, r, friendship.GetOutgoingRequestsResponse{
		Requests: resp.Requests,
		Cursor:   encodedCursor,
		Total:    resp.Total,
	})
}
