package api

import (
	"net/http"
	"strconv"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/chat/friendship/app/api/responses"
	"code.justin.tv/chat/friendship/app/api/utils"
	"code.justin.tv/chat/friendship/app/internal"
	friendship "code.justin.tv/chat/friendship/client"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
)

func (s *Server) listFriendsV0(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	required, err := gojiplus.FetchRequiredURLParams(ctx, "user_id")
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}
	userID := required["user_id"]

	maxFriendsLimit := internal.MaxFriends(userID)
	limit, err := requestLimit(r, maxFriendsLimit, maxFriendsLimit)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	cursor := requestCursor(r)

	sortDirection, err := requestSort(r, friendship.DescendingDir)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	resp, err := s.backend.GetFriends(ctx, userID, limit, cursor, sortDirection)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, utils.ErrorStatus(err))
		return
	}

	gojiplus.ServeJSON(w, r, responses.FriendListResponse{
		Friends: resp.Friends,
		Cursor:  resp.Cursor,
		Total:   resp.Total,
	})
}

func requestLimit(r *http.Request, defaultLimit, maxLimit int) (int, error) {
	limitStr := r.FormValue("limit")
	if limitStr == "" {
		return defaultLimit, nil
	}

	limit, err := strconv.Atoi(limitStr)
	if err != nil {
		return 0, errx.New(err)
	} else if limit <= 0 {
		return 0, errx.New("invalid limit query param", errx.Fields{"limit_query": limit})
	}

	if limit > maxLimit {
		limit = maxLimit
	}
	return limit, nil
}

func requestCursor(r *http.Request) string {
	return r.FormValue("cursor")
}

func requestSort(r *http.Request, defaultDir string) (string, error) {
	direction := r.FormValue("sort")
	if direction == "" {
		return defaultDir, nil
	}

	direction = strings.ToUpper(direction)
	if direction != friendship.AscendingDir && direction != friendship.DescendingDir {
		return "", errx.New("invalid sort query param", errx.Fields{"sort_query": direction})
	}
	return direction, nil
}
