package api

import (
	"fmt"
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/backend/backendtest"
)

func TestListFriends(t *testing.T) {
	t.Run("When finding friends", func(t *testing.T) {
		server, b := serverWithTestBackend(t)
		mockFriends := b.Friends.(*backendtest.CohesionClient)

		t.Run("should fail when the limit param is invalid", func(t *testing.T) {
			path := fmt.Sprintf("/users/%s/friends?limit=0", backend.BobID)

			w := doTestHTTP(t, server, "GET", path)
			assert.Equal(t, http.StatusBadRequest, w.Code)
			assert.Contains(t, w.Body.String(), http.StatusText(w.Code))
		})

		t.Run("should fail when the sort param is invalid", func(t *testing.T) {
			path := fmt.Sprintf("/users/%s/friends?sort=foobar", backend.BobID)

			w := doTestHTTP(t, server, "GET", path)
			assert.Equal(t, http.StatusBadRequest, w.Code)
			assert.Contains(t, w.Body.String(), http.StatusText(w.Code))
		})

		t.Run("should call the appropriate Cohesion client functions", func(t *testing.T) {
			path := fmt.Sprintf("/users/%s/friends", backend.AliceID)

			doTestHTTP(t, server, "GET", path)
			assert.Equal(t, 2, len(mockFriends.GetCalls()))

			calls := mockFriends.GetCalls()
			listCall := calls[0]
			expected := backendtest.Call{
				Name:       "ListAssoc",
				E1:         backend.AliceID,
				Kind:       backend.FriendsWith,
				EntityKind: backend.User,
			}
			validateCall(t, listCall, expected)

			countCall := calls[1]
			expected = backendtest.Call{
				Name:       "CountAssoc",
				E1:         backend.AliceID,
				Kind:       backend.FriendsWith,
				EntityKind: backend.User,
			}
			validateCall(t, countCall, expected)
		})
	})
}
