package api

import (
	"fmt"
	"net/http"
	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/redis"
)

func TestResetNotifications(t *testing.T) {
	b := backend.NewBackendWithMocks()
	s, err := NewServer(b, &statsd.NoopClient{})
	assert.Nil(t, err)

	mockRedis := b.Redis.(*redis.MockRedis)
	mockRedis.On("Del", mock.Anything, []string{"incoming_request:" + backend.BobID}).Return(1, nil).Once()
	mockRedis.On("Del", mock.Anything, []string{"notifications:" + backend.BobID}).Return(1, nil).Once()

	test := testutils.APITest{
		Method:         "DELETE",
		URL:            fmt.Sprintf("http://localhost:8000/users/%s/friends/notifications", backend.BobID),
		Headers:        map[string]string{"X-Forwarded-Proto": "https"},
		ExpectedStatus: http.StatusNoContent,
	}
	testutils.RunTest(t, s, test)
}

func TestGetNotifications(t *testing.T) {
	b := backend.NewBackendWithMocks()
	s, err := NewServer(b, &statsd.NoopClient{})
	assert.Nil(t, err)

	mockRedis := b.Redis.(*redis.MockRedis)
	mockRedis.On("Scard", mock.Anything, "incoming_request:"+backend.BobID).Return(1, nil)

	test := testutils.APITest{
		Method:         "GET",
		URL:            fmt.Sprintf("http://localhost:8000/users/%s/friends/notifications", backend.BobID),
		Headers:        map[string]string{"X-Forwarded-Proto": "https"},
		ExpectedStatus: http.StatusOK,
		Expected: map[string]interface{}{
			"count": float64(1),
		},
	}
	testutils.RunTest(t, s, test)
}
