package recommendations

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/url"
	"strconv"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
)

const defaultTimingXactName = "recommendations"

// Client represents a recommendations client
type Client interface {
	// GetRecommendations returns recommendations of all content types
	GetRecommendations(ctx context.Context, params *GetRecommendationsParams, reqOpts *twitchclient.ReqOpts) (*Recommendations, error)

	// GetTrending returns a list of trending videos
	GetTrending(ctx context.Context, params *GetTrendingParams, reqOpts *twitchclient.ReqOpts) (*Videos, error)

	// GetRecommendationsForDevice gets a list of recommendations for a device
	GetRecommendationsForDevice(ctx context.Context, params *GetRecommendationsForDeviceParams, reqOpts *twitchclient.ReqOpts) (*VodIDs, error)

	// GetRecommendationsForUser gets a list of recommendations for a device
	GetRecommendationsForUser(ctx context.Context, params *GetRecommendationsForUserParams, reqOpts *twitchclient.ReqOpts) (*VodIDs, error)

	// GetVodRecommendationsForDevice gets a list of recommendations for a device
	GetVodRecommendationsForDevice(ctx context.Context, params *GetVodRecommendationsForDeviceParams, reqOpts *twitchclient.ReqOpts) (*RecommendedVodIDs, error)

	// GetSocialRecommendationsForUser gets a list of social recommendations for a user
	GetSocialRecommendationsForUser(ctx context.Context, params *GetSocialRecommendationsForUserParams, reqOpts *twitchclient.ReqOpts) (*SocialRecommendedStreamList, error)

	// HasRecommendedStreams returns whether or not the channel has similar channel recommendations
	HasRecommendedStreams(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*RecommendedStreamsMetadata, error)

	// GetRecommended returns recommended streams for a channel
	GetRecommendedStreams(ctx context.Context, params *GetRecommendedStreamsParams, reqOpts *twitchclient.ReqOpts) ([]SimilarStreams, error)

	// GetOnboardingChanels returns recommended onboarding channels for a user and games
	GetOnboardingChannels(ctx context.Context, params *GetOnboardingChannelsParams, reqOpts *twitchclient.ReqOpts) (*OnboardingChannels, error)

	// GetFavoriteChannels returns similar channels for a channel
	GetFavoriteChannels(ctx context.Context, params *GetFavoriteChannelsParams, reqOpts *twitchclient.ReqOpts) ([]string, error)

	// GetUnfilteredFavoriteChannels returns unfiltered favorite channels
	GetUnfilteredFavoriteChannels(ctx context.Context, params *GetUnfilteredFavoriteChannelsParams, reqOpts *twitchclient.ReqOpts) ([]string, error)

	// GetFavoriteGames returns favorite games
	GetFavoriteGames(ctx context.Context, params *GetFavoriteGamesParams, reqOpts *twitchclient.ReqOpts) ([]string, error)

	// GetSimilarChannels returns similar channels for a channel
	GetSimilarChannels(ctx context.Context, params *GetSimilarChannelsParams, reqOpts *twitchclient.ReqOpts) ([]string, error)

	// GetSimilarVideos returns similar vods for a vodID
	GetSimilarVideos(ctx context.Context, params *GetSimilarVideosParams, reqOpts *twitchclient.ReqOpts) ([]string, error)

	// GetSimilarClips returns similar clips for a clip
	GetSimilarClips(ctx context.Context, params *GetSimilarClipsParams, reqOpts *twitchclient.ReqOpts) ([]string, error)

	// GetRealtimeClips returns realtime clips
	GetRealtimeClips(ctx context.Context, params *GetRealtimeClipsParams, reqOpts *twitchclient.ReqOpts) ([]RealtimeClip, error)
}

// canonical implementation of Client
type clientImpl struct {
	twitchclient.Client
}

// NewClient returns an object with the canonical implementation of Client
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &clientImpl{twitchClient}, err
}

func (c *clientImpl) GetRecommendations(ctx context.Context, params *GetRecommendationsParams, reqOpts *twitchclient.ReqOpts) (*Recommendations, error) {
	body, _ := json.Marshal(params.Context)
	u := url.URL{
		Path:     "/recs",
		RawQuery: fmt.Sprintf("user_id=%s", params.UserID),
	}

	req, err := c.NewRequest("POST", u.String(), bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_recommendations",
		StatSampleRate: 0.1,
	})
	var recs Recommendations
	_, err = c.DoJSON(ctx, &recs, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return &recs, err
}

func (c *clientImpl) GetTrending(ctx context.Context, params *GetTrendingParams, reqOpts *twitchclient.ReqOpts) (*Videos, error) {
	values := url.Values{
		"game":     params.Games,
		"language": params.Languages,
		"limit":    {strconv.Itoa(params.Limit)},
		"offset":   {strconv.Itoa(params.Offset)},
	}

	u := url.URL{
		Path:     "/trending",
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_trending_videos",
		StatSampleRate: 0.1,
	})
	var videos Videos
	_, err = c.DoJSON(ctx, &videos, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return &videos, err
}

func (c *clientImpl) GetRecommendationsForDevice(ctx context.Context, params *GetRecommendationsForDeviceParams, reqOpts *twitchclient.ReqOpts) (*VodIDs, error) {
	values := url.Values{
		"limit": {strconv.Itoa(params.Limit)},
		"group": {params.Group},
	}

	u := url.URL{
		Path:     fmt.Sprintf("/vods/devices/%s", params.DeviceID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_device_recommendations",
		StatSampleRate: 0.1,
	})
	var videos VodIDs
	_, err = c.DoJSON(ctx, &videos, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return &videos, err
}

func (c *clientImpl) GetRecommendationsForUser(ctx context.Context, params *GetRecommendationsForUserParams, reqOpts *twitchclient.ReqOpts) (*VodIDs, error) {
	values := url.Values{
		"limit": {strconv.Itoa(params.Limit)},
		"group": {params.Group},
	}

	u := url.URL{
		Path:     fmt.Sprintf("/vods/users/%s", params.UserID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_user_recommendations",
		StatSampleRate: 0.1,
	})
	var videos VodIDs
	_, err = c.DoJSON(ctx, &videos, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return &videos, err
}

func (c *clientImpl) GetVodRecommendationsForDevice(ctx context.Context, params *GetVodRecommendationsForDeviceParams, reqOpts *twitchclient.ReqOpts) (*RecommendedVodIDs, error) {
	values := url.Values{
		"limit":    {strconv.Itoa(params.Limit)},
		"offset":   {strconv.Itoa(params.Offset)},
		"group":    {params.Group},
		"language": params.Languages,
	}

	u := url.URL{
		Path:     fmt.Sprintf("/vods/devices/%s", params.DeviceID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_device_recommendations",
		StatSampleRate: 0.1,
	})
	var vodIDs RecommendedVodIDs
	_, err = c.DoJSON(ctx, &vodIDs, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return &vodIDs, err
}

func (c *clientImpl) GetSocialRecommendationsForUser(ctx context.Context, params *GetSocialRecommendationsForUserParams, reqOpts *twitchclient.ReqOpts) (*SocialRecommendedStreamList, error) {
	values := url.Values{
		"limit": {strconv.Itoa(params.Limit)},
	}

	u := url.URL{
		Path:     fmt.Sprintf("/streams/social/users/%s", params.UserID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_social_recommendations",
		StatSampleRate: 0.1,
	})
	var socialRecommendations SocialRecommendedStreamList
	_, err = c.DoJSON(ctx, &socialRecommendations, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return &socialRecommendations, err
}

func (c *clientImpl) HasRecommendedStreams(ctx context.Context, channelID string, reqOpts *twitchclient.ReqOpts) (*RecommendedStreamsMetadata, error) {
	req, err := c.NewRequest("GET", fmt.Sprintf("/streams/recommended/%v/exists", channelID), nil)
	if err != nil {
		return nil, err
	}

	var metadata RecommendedStreamsMetadata
	_, err = c.DoJSON(ctx, &metadata, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.has_recommended_streams",
		StatSampleRate: 0.1,
	}))

	return &metadata, err
}

func (c *clientImpl) GetRecommendedStreams(ctx context.Context, params *GetRecommendedStreamsParams, reqOpts *twitchclient.ReqOpts) ([]SimilarStreams, error) {
	values := url.Values{
		"limit":  {strconv.Itoa(params.Limit)},
		"offset": {strconv.Itoa(params.Offset)},
		"group":  {params.Group},
	}
	u := url.URL{
		Path:     fmt.Sprintf("/streams/recommended/%v", params.ChannelID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	var channels []SimilarStreams
	_, err = c.DoJSON(ctx, &channels, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_recommended_streams",
		StatSampleRate: 0.1,
	}))

	return channels, err
}

func (c *clientImpl) GetOnboardingChannels(ctx context.Context, params *GetOnboardingChannelsParams, reqOpts *twitchclient.ReqOpts) (*OnboardingChannels, error) {
	withTopClip := "false"
	if params.WithTopClip == true {
		withTopClip = "true"
	}
	values := url.Values{
		"game_ids":      {strings.Join(params.GameIDs, ",")},
		"language":      {params.Language},
		"with_top_clip": {withTopClip},
	}
	u := url.URL{
		Path:     fmt.Sprintf("/channels/onboarding/%v", params.UserID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	var channels map[string][]OnboardingStream
	_, err = c.DoJSON(ctx, &channels, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_onboarding_channels",
		StatSampleRate: 0.1,
	}))

	return &OnboardingChannels{Channels: channels}, err
}

func (c *clientImpl) GetFavoriteChannels(ctx context.Context, params *GetFavoriteChannelsParams, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	values := url.Values{
		"limit":  {strconv.Itoa(params.Limit)},
		"offset": {strconv.Itoa(params.Offset)},
	}
	u := url.URL{
		Path:     fmt.Sprintf("/channels/favorite/%s", params.ChannelID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	var channels []string
	_, err = c.DoJSON(ctx, &channels, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_favorite_channels",
		StatSampleRate: 0.1,
	}))

	return channels, err
}

func (c *clientImpl) GetFavoriteGames(ctx context.Context, params *GetFavoriteGamesParams, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	values := url.Values{
		"limit":  {strconv.Itoa(params.Limit)},
		"offset": {strconv.Itoa(params.Offset)},
	}
	u := url.URL{
		Path:     fmt.Sprintf("/games/favorite/%s", params.DeviceID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	var games []string
	_, err = c.DoJSON(ctx, &games, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_favorite_games",
		StatSampleRate: 0.1,
	}))

	return games, err
}

func (c *clientImpl) GetUnfilteredFavoriteChannels(ctx context.Context, params *GetUnfilteredFavoriteChannelsParams, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	values := url.Values{
		"limit":  {strconv.Itoa(params.Limit)},
		"offset": {strconv.Itoa(params.Offset)},
	}
	u := url.URL{
		Path:     fmt.Sprintf("/channels/unfiltered-favorite/%s", params.ChannelID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	var channels []string
	_, err = c.DoJSON(ctx, &channels, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_unfiltered_favorite_channels",
		StatSampleRate: 0.1,
	}))

	return channels, err
}

func (c *clientImpl) GetSimilarChannels(ctx context.Context, params *GetSimilarChannelsParams, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	values := url.Values{
		"limit":  {strconv.Itoa(params.Limit)},
		"offset": {strconv.Itoa(params.Offset)},
	}

	u := url.URL{
		Path:     fmt.Sprintf("/channels/similar/%s", params.ChannelID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	var channels []string
	_, err = c.DoJSON(ctx, &channels, req, twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_similar_channels",
		StatSampleRate: 0.1,
	}))

	return channels, err
}

func (c *clientImpl) GetSimilarVideos(ctx context.Context, params *GetSimilarVideosParams, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	values := url.Values{
		"limit":  {strconv.Itoa(params.Limit)},
		"offset": {strconv.Itoa(params.Offset)},
	}

	u := url.URL{
		Path:     fmt.Sprintf("/videos/similar/%s", params.VodID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_similar_videos",
		StatSampleRate: 0.1,
	})
	var videoIDS []string
	_, err = c.DoJSON(ctx, &videoIDS, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return videoIDS, err
}

func (c *clientImpl) GetSimilarClips(ctx context.Context, params *GetSimilarClipsParams, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	values := url.Values{
		"limit":  {strconv.Itoa(params.Limit)},
		"offset": {strconv.Itoa(params.Offset)},
	}

	u := url.URL{
		Path:     fmt.Sprintf("/clips/similar/%s", params.ClipID),
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_similar_clips",
		StatSampleRate: 0.1,
	})
	var clipIDS []string
	_, err = c.DoJSON(ctx, &clipIDS, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return clipIDS, err
}

func (c *clientImpl) GetRealtimeClips(ctx context.Context, params *GetRealtimeClipsParams, reqOpts *twitchclient.ReqOpts) ([]RealtimeClip, error) {
	values := url.Values{
		"channel_id":           {params.ChannelID},
		"language":             {params.Language},
		"game":                 {params.Game},
		"algorithm_id":         {params.AlgorithmID},
		"created_within_hours": {strconv.Itoa(params.CreatedWithinHours)},
		"limit":                {strconv.Itoa(params.Limit)},
		"offset":               {strconv.Itoa(params.Offset)},
	}

	u := url.URL{
		Path:     "/clips/realtime",
		RawQuery: values.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.recommendations.get_realtime_clips",
		StatSampleRate: 0.1,
	})
	var clips []RealtimeClip
	_, err = c.DoJSON(ctx, &clips, req, twitchclient.MergeReqOpts(reqOpts, combinedReqOpts))

	return clips, err
}
