package recommendations

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/http/httptest"
	"testing"

	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"

	. "github.com/smartystreets/goconvey/convey"
)

const (
	recommendationsResponseFile           = "testdata/recommendations_response.json"
	trendingResponseFile                  = "testdata/trending_response.json"
	userRecommendationsResponseFile       = "testdata/user_recs_response.json"
	userSocialRecommendationsResponseFile = "testdata/user_social_recs_response.json"
	deviceRecommendationsResponseFile     = "testdata/device_recs_response.json"
	vodDeviceRecommendationsResponseFile  = "testdata/vod_device_recs_response.json"
	recommendedStreamsResponseFile        = "testdata/recommended_streams_response.json"
	onboardingChannelsResponseFile        = "testdata/onboarding_channels_response.json"
)

func readTestFile(name string) []byte {
	response, err := ioutil.ReadFile(name)
	if err != nil {
		log.Fatalf("Error reading file %s: %v", name, err)
	}
	return response
}

func TestGetRecommendations(t *testing.T) {
	recsResponse := readTestFile(recommendationsResponseFile)

	userID := "asda"
	Convey("when calling GetRecommendations", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "POST" && r.URL.Path == "/recs" {
				q := r.URL.Query()
				if q.Get("user_id") != userID {
					t.Fatal("wrong user id")
				}

				var params map[string]interface{}
				dec := json.NewDecoder(r.Body)
				err := dec.Decode(&params)
				if err != nil {
					t.Fatal("bad body")
				}

				if params["foo"] != "bar" {
					t.Fatal("bad body field")
				}

				fmt.Fprintln(w, string(recsResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		recs, err := r.GetRecommendations(context.Background(), &GetRecommendationsParams{
			UserID: userID,
			Context: map[string]interface{}{
				"foo": "bar",
			},
		}, nil)

		So(err, ShouldBeNil)
		So(recs, ShouldNotBeNil)
		So(recs.GenerationID, ShouldEqual, "foo")
		So(len(recs.Recommendations), ShouldEqual, 3)
		So(recs.Recommendations[0].Kind, ShouldEqual, "clip")
		So(recs.Recommendations[1].Kind, ShouldEqual, "stream")
		So(recs.Recommendations[2].Kind, ShouldEqual, "vod")
	})
}

func TestGetTrending(t *testing.T) {
	trendingResponse := readTestFile(trendingResponseFile)

	Convey("when calling GetTrending", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/trending" {
				q := r.URL.Query()
				// So doesn't work here
				if !(len(q["language"]) == 4 && q["language"][0] == "l" && q["language"][1] == "a" && q["language"][2] == "n" && q["language"][3] == "g") {
					t.Fatal("language parameter not sent to URL")
				}
				if !(len(q["game"]) == 4 && q["game"][0] == "g" && q["game"][1] == "a" && q["game"][2] == "m" && q["game"][3] == "e") {
					t.Fatal("game parameter not sent to URL")
				}
				fmt.Fprintln(w, string(trendingResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		videos, err := r.GetTrending(context.Background(), &GetTrendingParams{
			Limit:     10,
			Offset:    0,
			Languages: []string{"l", "a", "n", "g"},
			Games:     []string{"g", "a", "m", "e"},
		}, nil)

		So(err, ShouldBeNil)
		So(videos, ShouldNotBeNil)
		So(len(videos.Vods), ShouldEqual, 10)
		So(videos.Vods[0].ID, ShouldEqual, "104480373")
		So(videos.Offset, ShouldEqual, 32)
	})
}

func TestGetRecommendationsForDevice(t *testing.T) {
	deviceResponse := readTestFile(deviceRecommendationsResponseFile)

	Convey("when calling GetRecommendationsForDevice", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/vods/devices/abc123" {
				q := r.URL.Query()
				if q["group"][0] != "a" {
					t.Fatal("group parameter not sent to URL")
				}
				if q["limit"][0] != "10" {
					t.Fatal("limit parameter not sent to URL")
				}
				fmt.Fprintln(w, string(deviceResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		vodIDs, err := r.GetRecommendationsForDevice(context.Background(), &GetRecommendationsForDeviceParams{
			Limit:    10,
			DeviceID: "abc123",
			Group:    "a",
		}, nil)

		So(err, ShouldBeNil)
		So(vodIDs, ShouldNotBeNil)
		So(len(vodIDs.IDs), ShouldEqual, 3)
		So(vodIDs.IDs[0], ShouldEqual, "105329230")
	})
}

func TestGetRecommendationsForUser(t *testing.T) {
	userResponse := readTestFile(userRecommendationsResponseFile)

	Convey("when calling GetRecommendationsForUser", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/vods/users/twitch" {
				q := r.URL.Query()
				if q["group"][0] != "a" {
					t.Fatal("group parameter not sent to URL")
				}
				if q["limit"][0] != "10" {
					t.Fatal("limit parameter not sent to URL")
				}
				fmt.Fprintln(w, string(userResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		vodIDs, err := r.GetRecommendationsForUser(context.Background(), &GetRecommendationsForUserParams{
			Limit:  10,
			UserID: "twitch",
			Group:  "a",
		}, nil)

		So(err, ShouldBeNil)
		So(vodIDs, ShouldNotBeNil)
		So(len(vodIDs.IDs), ShouldEqual, 3)
		So(vodIDs.IDs[0], ShouldEqual, "105411851")
	})
}

func TestGetVodRecommendationsForDevice(t *testing.T) {
	Convey("when calling GetVodRecommendationsForDevice", t, func() {
		deviceResponse := readTestFile(vodDeviceRecommendationsResponseFile)
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/vods/devices/abc123" {
				q := r.URL.Query()
				if q["group"][0] != "a" {
					t.Fatal("group parameter not sent to URL")
				}
				if q["limit"][0] != "10" {
					t.Fatal("limit parameter not sent to URL")
				}
				if q["offset"][0] != "2" {
					t.Fatal("offset parameter not sent to URL")
				}
				if !(len(q["language"]) == 2 && q["language"][0] == "l" && q["language"][1] == "a") {
					t.Fatal("language parameter not sent to URL")
				}
				fmt.Fprintln(w, string(deviceResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		recommendedVods, err := r.GetVodRecommendationsForDevice(context.Background(), &GetVodRecommendationsForDeviceParams{
			Limit:     10,
			Offset:    2,
			DeviceID:  "abc123",
			Group:     "a",
			Languages: []string{"l", "a"},
		}, nil)

		So(err, ShouldBeNil)
		So(recommendedVods, ShouldNotBeNil)
		So(len(recommendedVods.VodIDs), ShouldEqual, 3)
		So(recommendedVods.VodIDs[0], ShouldEqual, "105329230")
		So(recommendedVods.VodIDs[1], ShouldEqual, "104827967")
		So(recommendedVods.VodIDs[2], ShouldEqual, "105202225")
		So(recommendedVods.RecommendationSetID, ShouldEqual, "123455")
	})

	Convey("when calling GetVodRecommendationsForDevice with missing parameters", t, func() {
		deviceResponse := readTestFile(vodDeviceRecommendationsResponseFile)
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/vods/devices/abc123" {
				q := r.URL.Query()
				if q["group"][0] != "" {
					t.Fatal("group parameter sent to URL")
				}
				if q["limit"][0] != "0" {
					t.Fatal("limit parameter sent to URL")
				}
				if q["offset"][0] != "0" {
					t.Fatal("offset parameter sent to URL")
				}
				if len(q["language"]) != 0 {
					t.Fatal("language parameter sent to URL")
				}
				fmt.Fprintln(w, string(deviceResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		recommendedVods, err := r.GetVodRecommendationsForDevice(context.Background(), &GetVodRecommendationsForDeviceParams{
			DeviceID: "abc123",
		}, nil)

		So(err, ShouldBeNil)
		So(recommendedVods, ShouldNotBeNil)
		So(len(recommendedVods.VodIDs), ShouldEqual, 3)
		So(recommendedVods.VodIDs[0], ShouldEqual, "105329230")
		So(recommendedVods.VodIDs[1], ShouldEqual, "104827967")
		So(recommendedVods.VodIDs[2], ShouldEqual, "105202225")
		So(recommendedVods.RecommendationSetID, ShouldEqual, "123455")
	})
}

func TestGetSocialRecommendationsForUser(t *testing.T) {
	userResponse := readTestFile(userSocialRecommendationsResponseFile)

	Convey("when calling GetSocialRecommendationsForUser", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams/social/users/twitch" {
				q := r.URL.Query()
				if q["limit"][0] != "10" {
					t.Fatal("limit parameter not sent to URL")
				}
				fmt.Fprintln(w, string(userResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		streamList, err := r.GetSocialRecommendationsForUser(context.Background(), &GetSocialRecommendationsForUserParams{
			Limit:  10,
			UserID: "twitch",
		}, nil)

		So(err, ShouldBeNil)
		So(streamList, ShouldNotBeNil)
		So(len(streamList.RecommendedStreams), ShouldEqual, 2)
		So(streamList.RecommendedStreams[0].FriendID, ShouldEqual, "101090316")
		So(streamList.RecommendedStreams[0].Stream.ID, ShouldEqual, "24206368560")
		So(streamList.RecommendedStreams[0].Stream.ChannelID, ShouldEqual, "24692015")
		So(streamList.RecommendedStreams[0].Stream.Login, ShouldEqual, "test_overwatch")
		So(streamList.RecommendedStreams[0].Stream.Game, ShouldEqual, "Overwatch")
		So(streamList.RecommendedStreams[1].Stream.AverageFPS, ShouldEqual, 61.9732275657)
		So(streamList.RecommendedStreams[1].Stream.IsPlaylist, ShouldEqual, false)
		So(streamList.RecommendedStreams[1].Stream.Viewers, ShouldEqual, 116)
		So(streamList.RecommendedStreams[1].Stream.Delay, ShouldEqual, 0)
		So(streamList.RecommendedStreams[1].Stream.VideoHeight, ShouldEqual, 810)
		So(streamList.RecommendedStreams[1].Stream.CreatedAt, ShouldEqual, "2017-01-12T20:30:50Z")
	})
}

func TestGetRecommendedStreams(t *testing.T) {
	streamsResponse := readTestFile(recommendedStreamsResponseFile)

	Convey("when calling GetRecommendedStreams", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/streams/recommended/123" {
				fmt.Fprintln(w, string(streamsResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		recommendedStreams, err := r.GetRecommendedStreams(context.Background(), &GetRecommendedStreamsParams{
			ChannelID: "123",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(err, ShouldBeNil)
		So(len(recommendedStreams), ShouldEqual, 1)
		So(recommendedStreams[0].ChannelID, ShouldEqual, "1234")
		So(len(recommendedStreams[0].SimilarStreams), ShouldEqual, 1)
		So(recommendedStreams[0].SimilarStreams[0].ChannelID, ShouldEqual, "1234")
		So(recommendedStreams[0].SimilarStreams[0].Login, ShouldEqual, "esl_csgo_ru")
		So(recommendedStreams[0].SimilarStreams[0].Viewers, ShouldEqual, 1337)
	})
}

func TestGetOnboardingChannels(t *testing.T) {
	streamsResponse := readTestFile(onboardingChannelsResponseFile)

	Convey("when calling GetOnboardingChannels", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/channels/onboarding/123" {
				fmt.Fprintln(w, string(streamsResponse))
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		onboardingchannels, err := r.GetOnboardingChannels(context.Background(), &GetOnboardingChannelsParams{
			UserID:      "123",
			GameIDs:     []string{},
			WithTopClip: false,
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(err, ShouldBeNil)
		So(len(onboardingchannels.Channels), ShouldEqual, 1)
		So(len(onboardingchannels.Channels["493057"]), ShouldEqual, 3)
	})
}

func TestGetFavoriteChannels(t *testing.T) {
	Convey("when calling GetFavoriteChannels", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/channels/favorite/123" {
				fmt.Fprintln(w, `[
					"1",
					"2"
				]`)
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		channels, err := r.GetFavoriteChannels(context.Background(), &GetFavoriteChannelsParams{
			Limit:     10,
			Offset:    0,
			ChannelID: "123",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(len(channels), ShouldEqual, 2)
		So(channels[0], ShouldEqual, "1")
		So(channels[1], ShouldEqual, "2")
	})
}

func TestGetUnfilteredFavoriteChannels(t *testing.T) {
	Convey("when calling GetUnfilteredFavoriteChannels", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/channels/unfiltered-favorite/123" {
				fmt.Fprintln(w, `[
					"1",
					"2"
				]`)
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		channels, err := r.GetUnfilteredFavoriteChannels(context.Background(), &GetUnfilteredFavoriteChannelsParams{
			Limit:     10,
			Offset:    0,
			ChannelID: "123",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(len(channels), ShouldEqual, 2)
		So(channels[0], ShouldEqual, "1")
		So(channels[1], ShouldEqual, "2")
	})
}

func TestGetFavoriteGames(t *testing.T) {
	Convey("when calling GetFavoriteGames", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/games/favorite/123" {
				fmt.Fprintln(w, `[
					"1",
					"2"
				]`)
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		games, err := r.GetFavoriteGames(context.Background(), &GetFavoriteGamesParams{
			Limit:    10,
			Offset:   0,
			DeviceID: "123",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(len(games), ShouldEqual, 2)
		So(games[0], ShouldEqual, "1")
		So(games[1], ShouldEqual, "2")
	})
}

func TestGetSimilarChannels(t *testing.T) {
	Convey("when calling GetSimilarChannels", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/channels/similar/123" {
				fmt.Fprintln(w, `[
					"1",
					"2"
				]`)
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		channels, err := r.GetSimilarChannels(context.Background(), &GetSimilarChannelsParams{
			Limit:     10,
			Offset:    0,
			ChannelID: "123",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(len(channels), ShouldEqual, 2)
		So(channels[0], ShouldEqual, "1")
		So(channels[1], ShouldEqual, "2")
	})
}

func TestGetSimilarVideos(t *testing.T) {
	Convey("when calling GetSimilarVideos", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/videos/similar/123" {
				fmt.Fprintln(w, `[
					"foo",
					"bar"
				]`)
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		videos, err := r.GetSimilarVideos(context.Background(), &GetSimilarVideosParams{
			Limit:  10,
			Offset: 0,
			VodID:  "123",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(len(videos), ShouldEqual, 2)
		So(videos[0], ShouldEqual, "foo")
		So(videos[1], ShouldEqual, "bar")
	})
}

func TestGetSimilarClips(t *testing.T) {
	Convey("when calling GetSimilarClips", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/clips/similar/123" {
				fmt.Fprintln(w, `[
					"foo",
					"bar"
				]`)
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		clips, err := r.GetSimilarClips(context.Background(), &GetSimilarClipsParams{
			Limit:  10,
			Offset: 0,
			ClipID: "123",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(len(clips), ShouldEqual, 2)
		So(clips[0], ShouldEqual, "foo")
		So(clips[1], ShouldEqual, "bar")
	})
}

func TestGetRealtimeClips(t *testing.T) {
	Convey("when calling GetRealtimeClips", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.URL.Path == "/clips/realtime" && r.URL.Query().Get("game") == "what" {
				fmt.Fprintln(w, `[
					{"clip_id": "1", "clip_slug": "2", "channel_id": "3"}
				]`)
			}
		}))
		defer ts.Close()

		r, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		clips, err := r.GetRealtimeClips(context.Background(), &GetRealtimeClipsParams{
			Game: "what",
		}, nil)
		if err != nil {
			t.Fatal(err)
		}

		So(len(clips), ShouldEqual, 1)
		So(clips[0].ID, ShouldEqual, 1)
	})
}
