package main

import (
	"fmt"
	"log"
	"os"

	metricCollector "github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/afex/hystrix-go/plugins"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/discovery/recommendations/api"
	"code.justin.tv/discovery/recommendations/logic"
)

func main() {
	// To add app-specific config flags, you can call config.Register()
	// https://godoc.internal.justin.tv/code.justin.tv/common/config
	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	stats := config.Statsd()

	// Setup hystrix statsd
	hostName, err := os.Hostname()
	if err != nil {
		hostName = "unknown"
	}

	statsdPrefix := fmt.Sprintf("recommendations.%s.%s.hystrix", config.Environment(), hostName)

	c, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
		StatsdAddr: config.StatsdHostPort(),
		Prefix:     statsdPrefix,
	})
	if err == nil {
		metricCollector.Registry.Register(c.NewStatsdCollector)
	}

	logic := makeLogic()
	defer func() {
		_ = logic.Close()
	}()
	server, err := api.NewServer(logic, stats)
	if err != nil {
		log.Fatal(err)
	}

	log.Fatal(twitchhttp.ListenAndServe(server, twitchhttp.NewConfig()))
}

func makeLogic() logic.Logic {
	logicRes, err := logic.New()
	if err != nil {
		log.Fatalf("fail to initialize logic: %v", err)
	}
	return logicRes
}
