package ctxlog

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/feeds/log"

	goctx "context"

	"context"

	. "github.com/smartystreets/goconvey/convey"
)

var idKey = "hello"

type storeRequest struct {
	header string
	ctx    goctx.Context
}

func (s *storeRequest) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	s.header = r.Header.Get("X-Ctxlog-LogID")
	s.ctx = r.Context()
}

type CustomIDAdder struct {
	Next http.Handler
}

func (c *CustomIDAdder) ServeHTTP(rw http.ResponseWriter, req *http.Request) {
	ctx := req.Context()
	ctx = goctx.WithValue(ctx, &idKey, "123456")
	req = req.WithContext(ctx)
	c.Next.ServeHTTP(rw, req)
}

func TestHTTPHeaders(t *testing.T) {
	Convey("With a testing server", t, func() {
		store := &storeRequest{}
		logDebugCount := log.Counter{}
		logNormalCount := log.Counter{}

		logToNormal := log.MultiLogger([]log.Logger{&logNormalCount, t})
		logToDebug := log.MultiLogger([]log.Logger{&logDebugCount, t})

		ctxd := log.CtxDimensions{}
		elevateKey := "hi"
		elevatedLog := log.ElevatedLog{
			NormalLog: log.ContextLogger{
				Logger: logToNormal,
				Dims:   &ctxd,
			},
			DebugLog: log.ContextLogger{
				Logger: logToDebug,
				Dims:   &ctxd,
			},
			ElevateKey: elevateKey,
		}

		ctxl := Ctxlog{
			CtxDims:    &ctxd,
			ElevateKey: elevateKey,
		}

		ch := CtxHandler{
			Next:   store,
			Ctxlog: &ctxl,
			Logger: &elevatedLog,
		}

		cia := CustomIDAdder{
			Next: &ch,
		}

		ctx := goctx.Background()

		s := httptest.NewServer(&cia)
		Reset(func() {
			s.Close()
		})

		httpClient := &http.Client{}
		ld := LoggedDoer{
			Logger: &elevatedLog,
			C:      &ctxl,
			Client: httpClient,
		}

		var resp *http.Response
		var respErr error

		Convey("With nothing should set nothing", func() {
			req, err := WrapHTTPRequestWithCtxlog(&ctxl, http.NewRequest)(ctx, "GET", s.URL, nil)
			So(err, ShouldBeNil)
			Log(req.Context(), &elevatedLog, func() {
				resp, respErr = ld.Do(req)
			})
			So(respErr, ShouldBeNil)
			So(resp.Header.Get(ctxl.GetElevateLogHeader()), ShouldEqual, "")
			So(resp.Header.Get(ctxl.GetLogIDHeader()), ShouldNotEqual, "")
			So(store.header, ShouldEqual, resp.Header.Get(ctxl.GetLogIDHeader()))
			So(logDebugCount.Count, ShouldEqual, 6)
			So(logNormalCount.Count, ShouldEqual, 0)
		})

		Convey("With debug should debug", func() {
			req, err := WrapHTTPRequestWithCtxlog(&ctxl, http.NewRequest)(ctx, "GET", s.URL, nil)
			So(err, ShouldBeNil)
			req = req.WithContext(goctx.WithValue(req.Context(), elevateKey, true))
			Log(req.Context(), &elevatedLog, func() {
				resp, respErr = ld.Do(req)
			})
			So(err, ShouldBeNil)
			So(resp.Header.Get(ctxl.GetElevateLogHeader()), ShouldEqual, "")
			So(resp.Header.Get(ctxl.GetLogIDHeader()), ShouldNotEqual, "")
			So(store.header, ShouldEqual, "1")
			So(logDebugCount.Count, ShouldEqual, 0)
			So(logNormalCount.Count, ShouldEqual, 6)
		})

		Convey("With id should keep it", func() {
			req, err := WrapHTTPRequestWithCtxlog(&ctxl, http.NewRequest)(ctx, "GET", s.URL, nil)
			So(err, ShouldBeNil)
			req = req.WithContext(ctxd.Append(req.Context(), elevateKey, true, ctxl.GetLogIDHeader(), "1234"))
			req = req.WithContext(goctx.WithValue(req.Context(), elevateKey, true))

			Log(req.Context(), &elevatedLog, func() {
				resp, respErr = ld.Do(req)
			}, "msg", "a request")
			So(err, ShouldBeNil)
			So(resp.Header.Get(ctxl.GetElevateLogHeader()), ShouldEqual, "")
			So(resp.Header.Get(ctxl.GetLogIDHeader()), ShouldNotEqual, "")
			So(store.header, ShouldEqual, "1234")
			So(logDebugCount.Count, ShouldEqual, 0)
			So(logNormalCount.Count, ShouldEqual, 6)
		})

		Convey("With custom ID, should retain it", func() {
			ch.IDFetcher = func(ctx context.Context) string {
				idVal := ctx.Value(&idKey)
				if idVal != nil {
					return idVal.(string)
				}
				return ""
			}
			req, err := http.NewRequest("GET", s.URL, nil)
			So(err, ShouldBeNil)
			resp, err := httpClient.Do(req)
			So(err, ShouldBeNil)
			expectedDebugID := ch.IDFetcher(store.ctx)
			So(expectedDebugID, ShouldNotBeEmpty)
			So(resp.Header.Get(ctxl.GetLogIDHeader()), ShouldEqual, expectedDebugID)
			actualDebugID, _ := ctxl.ExtractDebugInfo(store.ctx)
			So(actualDebugID, ShouldEqual, expectedDebugID)
		})
	})
}
