package sfxstatsd

import (
	"errors"
	"sync"
	"sync/atomic"
	"time"

	"strings"

	"code.justin.tv/feeds/metrics/statsdim"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/signalfx/golib/datapoint"
	"github.com/signalfx/golib/sfxclient"
)

// TranslatedMetricsStore remembers statsd translations and stores them in a way signalfx can accept
type TranslatedMetricsStore struct {
	mu         sync.RWMutex
	inMap      map[string]*intMetric
	rollingMap map[string]*rollingMetric
}

// Statter implements statsd.Statter interface and sends to SignalFx
type Statter struct {
	StatSender
}

// SubStatter implements statsd.SubStatter interface and sends to SignalFx
type SubStatter struct {
	StatSender
}

// StatSender implements statsd.StatSender interface and sends to SignalFx
type StatSender struct {
	Store      *TranslatedMetricsStore `nilcheck:"ignore"`
	Translator *statsdim.StatsdTranslator
}

var _ statsd.StatSender = &StatSender{}
var _ statsd.Statter = &Statter{}
var _ statsd.SubStatter = &SubStatter{}

type intMetric struct {
	metricName string
	dims       map[string]string
	dpType     datapoint.MetricType
	val        int64
}

func (i *intMetric) Datapoint() *datapoint.Datapoint {
	return datapoint.New(i.metricName, i.dims, datapoint.NewIntValue(atomic.LoadInt64(&i.val)), i.dpType, time.Time{})
}

type rollingMetric struct {
	metricName string
	dims       map[string]string
	val        *sfxclient.RollingBucket
}

func (i *rollingMetric) Datapoints() []*datapoint.Datapoint {
	dp := i.val.Datapoints()
	ret := make([]*datapoint.Datapoint, 0, len(dp))
	for _, d := range dp {
		// Not very useful
		if strings.HasSuffix(d.Metric, ".sumsquare") {
			continue
		}
		ret = append(ret, d)
	}
	return ret
}

func (c *TranslatedMetricsStore) Datapoints() []*datapoint.Datapoint {
	c.mu.RLock()
	defer c.mu.RUnlock()
	ret := make([]*datapoint.Datapoint, 0, len(c.inMap)+len(c.rollingMap)*7)
	for _, i := range c.inMap {
		if i == nil {
			continue
		}
		ret = append(ret, i.Datapoint())
	}
	for _, r := range c.rollingMap {
		if r == nil {
			continue
		}
		ret = append(ret, r.Datapoints()...)
	}
	return ret
}

func (s *Statter) Close() error {
	return nil
}

func newSubStatter(k string, store *TranslatedMetricsStore, Translator *statsdim.StatsdTranslator) *SubStatter {
	x := &SubStatter{
		StatSender: StatSender{
			Store:      store,
			Translator: Translator.NewSubStatter(k),
		},
	}
	return x
}

func (s *Statter) NewSubStatter(k string) statsd.SubStatter {
	return newSubStatter(k, s.StatSender.Store, s.StatSender.Translator)
}

func (s *SubStatter) SetSamplerFunc(_ statsd.SamplerFunc) {
	// Ignored
}

func (s *SubStatter) NewSubStatter(k string) statsd.SubStatter {
	return newSubStatter(k, s.StatSender.Store, s.StatSender.Translator)
}

func (s *Statter) SetPrefix(k string) {
	s.StatSender.Translator.SetPrefix(k)
}

func (s *TranslatedMetricsStore) lookupRolling(cacheKey string, statsdValue string, Translator func(string) (string, map[string]string, error)) (*rollingMetric, error) {
	s.mu.RLock()
	current, exists := s.rollingMap[cacheKey]
	s.mu.RUnlock()
	if exists {
		if current == nil {
			return nil, errors.New("unable to find key " + cacheKey)
		}
		return current, nil
	}
	s.mu.Lock()
	defer s.mu.Unlock()
	if s.rollingMap == nil {
		s.rollingMap = make(map[string]*rollingMetric, 24)
	}
	current, exists = s.rollingMap[cacheKey]
	if exists {
		return current, nil
	}

	metricName, dims, err := Translator(statsdValue)
	if err != nil {
		s.rollingMap[cacheKey] = nil
		return nil, err
	}
	bucket := sfxclient.NewRollingBucket(metricName, dims)
	bucket.Quantiles = []float64{.5, .9, .99}
	ret := &rollingMetric{
		metricName: metricName,
		dims:       dims,
		val:        bucket,
	}
	s.rollingMap[cacheKey] = ret
	return ret, nil
}

func (s *TranslatedMetricsStore) lookupInt(cacheKey string, statsdValue string, Translator func(key string) (string, map[string]string, error), dpType datapoint.MetricType) (*intMetric, error) {
	s.mu.RLock()
	current, exists := s.inMap[cacheKey]
	s.mu.RUnlock()
	if exists {
		if current == nil {
			return nil, errors.New("unable to find key " + cacheKey)
		}
		return current, nil
	}

	s.mu.Lock()
	defer s.mu.Unlock()
	if s.inMap == nil {
		s.inMap = make(map[string]*intMetric, 24)
	}
	current, exists = s.inMap[cacheKey]
	if exists {
		return current, nil
	}
	metricName, dims, err := Translator(statsdValue)
	if err != nil {
		s.inMap[cacheKey] = nil
		return nil, err
	}
	ret := &intMetric{
		metricName: metricName,
		dims:       dims,
		dpType:     dpType,
	}
	s.inMap[cacheKey] = ret
	return ret, nil
}

func (s *StatSender) Inc(originalMetric string, value int64, r float32) error {
	m, err := s.Store.lookupInt(s.Translator.FullKey(originalMetric), originalMetric, s.Translator.Inc, datapoint.Counter)
	if err != nil {
		return err
	}
	atomic.AddInt64(&m.val, value)
	return nil
}

func (s *StatSender) Gauge(originalMetric string, value int64, r float32) error {
	m, err := s.Store.lookupInt(s.Translator.FullKey(originalMetric), originalMetric, s.Translator.Gauge, datapoint.Gauge)
	if err != nil {
		return err
	}
	atomic.StoreInt64(&m.val, value)
	return nil
}

func (s *StatSender) GaugeDelta(originalMetric string, value int64, r float32) error {
	m, err := s.Store.lookupInt(s.Translator.FullKey(originalMetric), originalMetric, s.Translator.Gauge, datapoint.Gauge)
	if err != nil {
		return err
	}
	atomic.AddInt64(&m.val, value)
	return nil
}

func (m *StatSender) Dec(originalMetric string, value int64, r float32) error {
	return m.Inc(originalMetric, -value, r)
}

func (s *StatSender) Set(originalMetric string, value string, r float32) error {
	return errors.New("unsupported")
}

func (s *StatSender) Raw(query string, value string, r float32) error {
	return errors.New("unsupported")
}

func (s *StatSender) SetSamplerFunc(sampler statsd.SamplerFunc) {
	// Not needed
}

func (s *StatSender) SetInt(originalMetric string, value int64, r float32) error {
	return errors.New("unsupported")
}

func (s *StatSender) TimingDuration(originalMetric string, value time.Duration, r float32) error {
	m, err := s.Store.lookupRolling(s.Translator.FullKey(originalMetric), originalMetric, s.Translator.Timing)
	if err != nil {
		return err
	}
	m.val.Add(value.Seconds())
	return nil
}

func (s *StatSender) Timing(originalMetric string, value int64, r float32) error {
	return s.TimingDuration(originalMetric, time.Duration(time.Millisecond.Nanoseconds()*value), r)
}
