package feedcache

import (
	"context"
	"encoding/json"
	"reflect"
	"time"

	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
)

type jsonDecoder struct{}

var defaultDecoder ValueDecoder = jsonDecoder{}

// Unmarshal converts stored data into a value
func (_ jsonDecoder) Unmarshal(data []byte, into interface{}) error {
	return json.Unmarshal(data, into)
}

// Marshal converts a value into []byte we can store
func (_ jsonDecoder) Marshal(val interface{}) ([]byte, error) {
	return json.Marshal(val)
}

// ValueDecoder controls how data is stored inside the cache backend.  It is usually JSON, but could
// also gzip the data
type ValueDecoder interface {
	Unmarshal(data []byte, into interface{}) error
	Marshal(val interface{}) ([]byte, error)
}

// CacheClientPool controls a pool of Cache clients that can store/get data
type CacheClientPool interface {
	GetClient(ctx context.Context) CacheClient
}

// CacheClient is a single connection that is returned (via Close) to the pool when finished
type CacheClient interface {
	Close(ctx context.Context) error
	Delete(ctx context.Context, key string) error
	Get(ctx context.Context, key string) ([]byte, error)
	Set(ctx context.Context, key string, value []byte, ttl time.Duration) error
}

// ObjectCache controls how to cache general data in Go
type ObjectCache struct {
	ClientPool   CacheClientPool
	Stats        *service_common.StatSender
	KeyPrefix    string
	KeyTTL       func(key string) time.Duration
	Log          *log.ElevatedLog
	ValueDecoder ValueDecoder `nilcheck:"ignore"`
}

func (r *ObjectCache) key(key string) string {
	return r.KeyPrefix + key
}

func (r *ObjectCache) keyTTL(key string) time.Duration {
	if r.KeyTTL == nil {
		return time.Duration(0)
	}
	return r.KeyTTL(key)
}

func (r *ObjectCache) valueDecoder() ValueDecoder {
	if r.ValueDecoder == nil {
		return defaultDecoder
	}
	return r.ValueDecoder
}

// Invalidate deletes a cached key
func (r *ObjectCache) Invalidate(ctx context.Context, key string) error {
	key = r.key(key)
	c := r.ClientPool.GetClient(ctx)
	defer func() {
		if err := c.Close(ctx); err != nil {
			r.Stats.IncC("close_err", 1, 1.0)
		}
	}()
	return c.Delete(ctx, key)
}

// ForceCached pushes an object into the cache, even if one exists
func (r *ObjectCache) ForceCached(ctx context.Context, key string, vals interface{}) error {
	c := r.ClientPool.GetClient(ctx)
	defer func() {
		if err := c.Close(ctx); err != nil {
			r.Stats.IncC("close_err", 1, 1.0)
		}
	}()
	key = r.key(key)
	return r.forceCachedWithClient(ctx, c, key, vals)
}

func (r *ObjectCache) forceCachedWithClient(ctx context.Context, c CacheClient, key string, vals interface{}) error {
	bytes, err := r.valueDecoder().Marshal(vals)
	if err != nil {
		r.Stats.IncC("cache.decode_err", 1, 1.0)
		return err
	}
	var ttl time.Duration
	if objectHasTTL, ok := vals.(objectWithTTL); ok {
		ttl = objectHasTTL.CacheTTL()
	} else {
		ttl = r.keyTTL(key)
	}
	if err := c.Set(ctx, key, bytes, ttl); err != nil {
		r.Stats.IncC("error", 1, 1.0)
		return err
	}
	return nil
}

// Cached will fetch a key into storeIntoPtr using callback if it doesn't exist
func (r *ObjectCache) Cached(ctx context.Context, key string, callback func() (interface{}, error), storeIntoPtr interface{}) error {
	r.Log.DebugCtx(ctx, "key", key, "cache.fetch")
	c := r.ClientPool.GetClient(ctx)
	defer func() {
		if err := c.Close(ctx); err != nil {
			r.Stats.IncC("close_err", 1, 1.0)
		}
	}()
	return r.cached(ctx, c, key, storeIntoPtr, callback)
}

// Ints is a Cached helper specifically for []int
func (r *ObjectCache) Ints(ctx context.Context, key string, callback func() ([]int64, error)) ([]int64, error) {
	var storeIntoPtr []int64
	err := r.Cached(ctx, key, func() (interface{}, error) {
		return callback()
	}, &storeIntoPtr)
	if err != nil {
		return nil, err
	}
	return storeIntoPtr, nil
}

type objectWithTTL interface {
	CacheTTL() time.Duration
}

func (r *ObjectCache) cached(ctx context.Context, c CacheClient, key string, storeInto interface{}, callback func() (interface{}, error)) error {
	key = r.key(key)
	existingKey, err := c.Get(ctx, key)
	if err == nil && existingKey != nil {
		if uerr := r.valueDecoder().Unmarshal(existingKey, storeInto); uerr != nil {
			return uerr
		}
		r.Stats.IncC("cache.hit", 1, 1.0)
		return nil
	}
	r.Stats.IncC("cache.miss", 1, 1.0)
	vals, err := callback()
	if err != nil {
		return err
	}
	rv := reflect.ValueOf(storeInto).Elem()
	if vals == nil {
		rv.Set(reflect.Zero(rv.Type()))
	} else {
		rv.Set(reflect.ValueOf(vals))
	}
	return r.forceCachedWithClient(ctx, c, key, vals)
}
