package feedsdynamo

import (
	"fmt"
	"strconv"

	"code.justin.tv/feeds/errors"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// AwsStrings parses required string values from dynamo attributes
func AwsStrings(fi map[string]*dynamodb.AttributeValue, stringKeys []string) (map[string]string, error) {
	ret := make(map[string]string, len(stringKeys))
	for _, s := range stringKeys {
		val, exists := fi[s]
		if !exists {
			return nil, errors.Errorf("unable to find key %s", s)
		}
		if val.S == nil {
			return nil, errors.Errorf("key %s isn't a string", s)
		}
		ret[s] = *val.S
	}
	return ret, nil
}

// AwsFloats parses required float values from dynamo attributes
func AwsFloats(fi map[string]*dynamodb.AttributeValue, floatKeys []string) (map[string]float64, error) {
	ret := make(map[string]float64, len(floatKeys))
	for _, s := range floatKeys {
		val, exists := fi[s]
		if !exists {
			return nil, errors.Errorf("unable to find key %s", s)
		}
		if val.N == nil {
			return nil, errors.Errorf("key %s isn't a number", s)
		}
		f, err := strconv.ParseFloat(*val.N, 64)
		if err != nil {
			return nil, errors.Wrap(err, fmt.Sprintf("value %s isn't a float: %s", s, *val.N))
		}
		ret[s] = f
	}
	return ret, nil
}

// AwsInts parses required int64 values from dynamo attributes
func AwsInts(fi map[string]*dynamodb.AttributeValue, intKeys []string) (map[string]int64, error) {
	ret := make(map[string]int64, len(intKeys))
	for _, s := range intKeys {
		val, exists := fi[s]
		if !exists {
			return nil, errors.Errorf("unable to find int key %s", s)
		}
		if val.N == nil {
			return nil, errors.Errorf("key %s isn't a number", s)
		}
		i, err := strconv.ParseInt(*val.N, 10, 64)
		if err != nil {
			return nil, errors.Wrap(err, fmt.Sprintf("value %s isn't an integer: %s", s, *val.N))
		}
		ret[s] = i
	}
	return ret, nil
}
