# XRay SDK

XRay Recorder SDK for Golang

## Code Snippets

**Configuration**

```go
import (
  "context"
  //This is made up and subject to change in the future
  "github.com/aws/xray"
  //Importing the plugins will enable them for xray
  _ "github.com/aws/xray/plugins/ec2"
  _ "github.com/aws/xray/plugins/beanstalk"
)

func init() {
  xray.Configure(xray.Config{
    DefaultName: "MyService",
    Host: "localhost", //default
    Port: 2000,        //default
  })
}
```

**Errors**

The `xray` package provides `Error`/`Errorf` helpers to create errors with valid stack traces. Errors generated by `github.com/pkg/errors` are also supported.

```go
func alwaysError() error {
  return xray.Error("always") // The stack trace will start at this line
}
```

If recovering from a panic, the `Panic`/`Panicf` helpers trim the stack for clearer debugging

```go
func returnAnErrorEvenIfItPanics() (err error) {
  defer func() {
    if p := recover(); p != nil {
      err = xray.Panicf("%v", p)
    }
  }

  panic("oh no") // The stack trace will start at this line
  return nil
}
```

**Capture**

```go
func criticalSection(ctx context.Context) {
  //this is an example of a subsegment
  xray.Capture(ctx, "MyService.criticalSection", func(ctx1 context.Context) error {
    var err error

    section.Lock()
    result := someLockedResource.Go()
    section.Unlock()

    xray.AddMetadata(ctx1, "ResourceResult", result)
  })
```

**HTTP Handler**

```go
func main() {
  http.Handle("/", xray.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
    w.Write([]byte("Hello!"))
  })))
  http.ListenAndServe(":8000", nil)
}
```

**HTTP Client**

```go
func getExample(ctx context.Context) ([]byte, error) {
    resp, err := ctxhttp.Get(ctx, xray.Client(nil), "http://example.com/")
    if err != nil {
      return nil, err
    }
    return ioutil.Readall(resp.Body)
}
```

**AWS**

```go
var dynamo *dynamodb.DynamoDB
func main() {
  dynamo = dynamodb.New(session.New())
  xray.AWS(dynamo.Client)
}

func listTablesWithContext(ctx context.Context) {
  output := dynamo.ListTablesWithContext(ctx, &dynamodb.ListTablesInput{})
  doSomething(output)
}
```

**SQL**

Any `db/sql` calls can be traced with XRay by replacing the `sql.Open` call with `xray.SQL`. It is recommended to use URLs instead of configuration strings if possible.

```go
func main() {
  db := xray.SQL("postgres", "postgres://user:password@host:port/db")
  row, _ := db.QueryRow("SELECT 1") // Use as normal
}
```
