// +build integration

package manager

import (
	"errors"
	"io/ioutil"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/inventory/consumedsecrets"
	"code.justin.tv/systems/sandstorm/queue"
	"code.justin.tv/systems/sandstorm/testutil"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/assert"
)

func nopLogger() *logrus.Logger {
	return &logrus.Logger{
		Out: ioutil.Discard,
	}
}

func createTestManager() (*Manager, error) {
	config, err := testutil.LoadTestConfig()
	if err != nil {
		return nil, err
	}
	if config == nil {
		return nil, errors.New("test config is nil")
	}

	awsConfig := &aws.Config{
		Region: aws.String(config.Sandstorm.Region),
	}
	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      config.Sandstorm.RoleArn,
		Client:       stsclient,
	}
	awsConfig.WithCredentials(credentials.NewCredentials(arp))
	return New(Config{
		AWSConfig:          awsConfig,
		TableName:          config.Sandstorm.TableName,
		KeyID:              config.Sandstorm.KeyID,
		Host:               "someHostName",
		InventoryRoleARN:   config.Sandstorm.InventoryRoleARN,
		InventoryStatusURL: config.Sandstorm.InventoryStatusURL,
		InventoryInterval:  1 * time.Second,
		ServiceName:        config.Sandstorm.ServiceName,
		Queue: queue.Config{
			TopicArn: "arn:aws:sns:us-west-2:734326455073:sandstorm-testing",
		},
		Logger: noopLogger,
	}), nil
}

func createTestConsumedSecretsClient(t *testing.T) (api consumedsecrets.API) {
	config, err := testutil.LoadTestConfig()
	if err != nil {
		t.Fatal(err)
	}

	awsConfig := &aws.Config{
		Region: aws.String(config.Sandstorm.Region),
	}

	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      config.Sandstorm.InventoryAdminRoleARN,
		Client:       stsclient,
	}
	awsConfig.WithCredentials(credentials.NewCredentials(arp))

	sess := session.New(awsConfig)

	api = consumedsecrets.New(
		&consumedsecrets.Config{
			TableName: "heartbeats-testing",
		},
		sess,
	)

	return
}

func TestE2EListenForUpdates(t *testing.T) {
	m, err := createTestManager()
	if err != nil {
		t.Fatal(err)
	}

	secretName := testutil.GetRandomSecretName()
	defer func() {
		err := m.Delete(secretName)
		assert.NoError(t, err)

		err = m.StopListeningForUpdates()
		assert.NoError(t, err)
	}()

	const (
		originalSecretValue = "plaintextSecret"
		patchedSecretValue  = "patchedPlaintextSecret"
	)

	err = m.Post(&Secret{
		Class:     ClassPublic,
		Name:      secretName,
		Plaintext: []byte(originalSecretValue),
	})
	if err != nil {
		t.Fatal(err)
	}

	err = m.ListenForUpdates()
	if err != nil {
		t.Fatal(err)
	}

	_, err = m.Get(secretName)
	if err != nil {
		t.Fatal(err)
	}

	t.Run("retrieve secret, secret is cached", func(t *testing.T) {
		cachedSecret := m.cache.Get(secretName)
		assert.NotNil(t, cachedSecret)
		assert.Equal(t, originalSecretValue, string(cachedSecret.Plaintext))
	})

	t.Run("rotate secret", func(t *testing.T) {
		err := m.Patch(&PatchInput{
			Name:      secretName,
			Plaintext: []byte(patchedSecretValue),
		})
		if err != nil {
			t.Fatal(err)
		}

		timeout := time.NewTimer(5 * time.Second)

		for {
			select {
			case <-timeout.C:
				t.Fatal("did not receive sqs message updating secret " + secretName)
			default:
			}

			cachedSecret := m.cache.Get(secretName)
			if cachedSecret == nil {
				break
			}
		}
	})
}
