/* global process, module, __dirname */
var path = require('path');
var converter = require('i18next-conv');
var fs = require('fs');

module.exports = function(grunt) {
  grunt.initConfig({
    localization: grunt.file.readJSON(path.resolve(__dirname, 'smartling.config.json'))
  });

  grunt.config.requires('localization');

  grunt.loadNpmTasks('grunt-smartling-sdk');

  var conf = grunt.config.get('localization');

  grunt.config.merge({
    smartling_get: { // eslint-disable-line camelcase
      options: {
        smartling: conf.options,
        operation: conf.operation.get,
        fileUriFunc() {
          return 'strings.pot';
        },
        destFileUriFunc() {
          return 'strings.po';
        },
        verbose: true
      },
      array: {
        locales: conf.locales,
        src:     path.resolve(__dirname, 'i18n/client/strings.pot'),
        dest:    path.resolve(__dirname, 'i18n/client/localization')
      }
    },
    smartling_upload: { // eslint-disable-line camelcase
      options: {
        smartling: conf.options,
        operation: conf.operation.upload,
        fileUriFunc(filepath) {
          return path.basename(filepath);
        }
      },
      one: {
        src: './i18n/client/strings.pot'
      }
    }
  });

  grunt.registerTask(
        'i18n',
        'Upload & retrieve translations for this project',
    [
      'smartling_upload',
      'smartling_get',
      'po-to-json'
    ]
    );

  grunt.registerTask('po-to-json', 'Task to convert .po files to .json', function() {
    var done = this.async();
    var save = (target) => {
      return (result) => {
        fs.writeFileSync(target, result);
      };
    };
    var locales = grunt.config.get('localization.locales');
    var options = {
      quiet: true,
      filter(gt, domain, callback) {
        grunt.log.write(`Converting locale ${domain}... `);

        var domainObject = gt.domains[gt._normalizeDomain(domain)];

        if (domain === 'en-US') {
          var translations = domainObject.translations[''];
          Object.keys(translations).forEach(function(key) {
            translations[key].msgstr = [ translations[key].msgid ];
            if (translations[key].hasOwnProperty('msgid_plural')) {
              translations[key].msgstr.push(translations[key].msgid_plural);
            }
          });
        }

        grunt.log.ok();

        callback(null, domainObject.translations);
      }
    };

    var localePromises = locales.map(function(locale) {
      var source = `i18n/client/localization/${locale}/strings.po`;
      var target = `i18n/client/localization/${locale}/strings.json`;
      var convertPromise = converter.gettextToI18next(locale, fs.readFileSync(source), options);

      return convertPromise.then(save(target, locale, source));
    });

    Promise.all(localePromises).then(done, done);
  });
};
