package clips

import (
	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "clips"
)

// Client is the interface to interact with the Clips APIs.
type Client interface {
	// DeleteClipBySlugV3 is an internal endpoint that allows a user to delete a
	// Clip given its slug. Note that only some users are allowed to delete a Clip.
	DeleteClipBySlugV3(context.Context, string, string, *twitchclient.ReqOpts) error

	// BatchDeleteClipsV3 is an internal endpoint to delete all clips based on a
	// list of slugs. Note that a list of DeletedClips is returned which only contains
	// the slug and id of a Clip.
	BatchDeleteClipsV3(ctx context.Context, userID string, slugs []string, opts *twitchclient.ReqOpts) (DeletedClips, error)

	// DeleteClipsByVODV3 is an internal endpoint to delete all clips from a vod.
	// Note that a list of DeletedClips is returned which only contains the slug
	// and id of a Clip.
	DeleteClipsByVODV3(ctx context.Context, userID, vodID string, opts *twitchclient.ReqOpts) (DeletedClips, error)

	// DeleteClipsByBroadcastV3 is an internal endpoint to delete all clips from
	// a broadcast. Note that a list of DeletedClips is returned which only contains
	// the slug and id of a Clip.
	DeleteClipsByBroadcastV3(ctx context.Context, userID, broadcastID string, opts *twitchclient.ReqOpts) (DeletedClips, error)

	ReportClipBySlugV3(ctx context.Context, userID, slug, reason, description string, opts *twitchclient.ReqOpts) error
	BanUserV3(ctx context.Context, channelID, requesterUserID, userToBanID string, isTemporary bool, opts *twitchclient.ReqOpts) error
	EditClipTitleV3(ctx context.Context, slug, userID, title string, opts *twitchclient.ReqOpts) (*Clip, error)
	GetClipInfoV3(ctx context.Context, slug string, params *GetClipInfoParams, opts *twitchclient.ReqOpts) (*Clip, error)
	GetTopClipsV3(ctx context.Context, params *GetTopClipsParams, opts *twitchclient.ReqOpts) (*PaginatedClipsResponse, error)
	GetMyClipsV3(ctx context.Context, params *GetMyClipsParams, opts *twitchclient.ReqOpts) (*PaginatedClipsResponse, error)
}

type client struct {
	twitchclient.Client
}

// NewClient takes a twitchclient.ClientConf and returns a client that implements
// the Clips Client interface.
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}
