package clipsv2

import (
	"errors"
	"fmt"
	"net/http"
	"net/url"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/video/clips-upload/client/utils"
	"golang.org/x/net/context"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "clips"
)

// Client in the interface a Clips client implements.
type Client interface {
	GetClip(ctx context.Context, slug string, params *GetClipParams, opts *twitchhttp.ReqOpts) (*Clip, error)
	GetClipStatus(ctx context.Context, slug string, opts *twitchhttp.ReqOpts) (*Status, error)
	GetTopClips(ctx context.Context, params *GetTopClipsParams, opt *twitchhttp.ReqOpts) (*PaginatedClipsResponse, error)
	GetClipCountByBroadcastID(ctx context.Context, broadcastID string, opts *twitchhttp.ReqOpts) (*ClipCount, error)
	GetClipsByBroadcastID(ctx context.Context, params *GetClipsByBroadcastIDParams, opts *twitchhttp.ReqOpts) (*PaginatedClipsResponse, error)
}

type client struct {
	twitchhttp.Client
}

type PaginatedClipsResponse struct {
	Clips  []*Clip `json:"clips"`
	Cursor string  `json:"cursor"`
}

// NewClient takes a twitchhttp.ClientConf and returns a client implementing the
// Clips client interface
func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchhttp.NewClient(conf)
	return &client{twitchClient}, err
}

func (c *client) GetClip(ctx context.Context, slug string, params *GetClipParams, opts *twitchhttp.ReqOpts) (*Clip, error) {
	if params == nil {
		params = NewGetClipParams()
	}

	req, err := c.NewRequest("GET", fmt.Sprintf("/api/v2/clips/%s?%s", slug, params.URLParamsString()), nil)
	if err != nil {
		return nil, err
	}

	clip := Clip{}
	combinedOpts := twitchhttp.MergeReqOpts(opts, twitchhttp.ReqOpts{
		StatName:       "service.clipsv2.get_clip",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.DoJSON(ctx, &clip, req, combinedOpts)
	if err != nil {
		if resp != nil {
			return nil, utils.ConstructSingleClipTwitchError(resp)
		}
		return nil, err
	}
	return &clip, nil
}

func (c *client) GetClipStatus(ctx context.Context, slug string, opts *twitchhttp.ReqOpts) (*Status, error) {
	req, err := c.NewRequest("GET", fmt.Sprintf("/api/v2/clips/%s/status", slug), nil)
	if err != nil {
		return nil, err
	}

	status := Status{}
	combinedOpts := twitchhttp.MergeReqOpts(opts, twitchhttp.ReqOpts{
		StatName:       "service.clipsv2.get_clip_status",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.DoJSON(ctx, &status, req, combinedOpts)
	if err != nil {
		if resp != nil {
			return nil, utils.ConstructSingleClipTwitchError(resp)
		}
		return nil, err
	}
	return &status, nil
}

func (c *client) GetTopClips(ctx context.Context, params *GetTopClipsParams, opts *twitchhttp.ReqOpts) (*PaginatedClipsResponse, error) {
	if params == nil {
		// instantiate params as the default params if none are passed in.
		params = NewGetTopClipsParams()
	}

	req, err := c.NewRequest("GET", fmt.Sprintf("/api/v2/clips/top?%s", params.URLParamsString()), nil)
	if err != nil {
		return nil, err
	}

	topClipsResponse := PaginatedClipsResponse{}
	combinedOpts := twitchhttp.MergeReqOpts(opts, twitchhttp.ReqOpts{
		StatName:       "service.clipsv2.get_top_clips",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.DoJSON(ctx, &topClipsResponse, req, combinedOpts)
	if err != nil {
		if resp != nil && resp.StatusCode == 404 {
			return nil, &twitchhttp.Error{StatusCode: http.StatusNotFound, Message: "no clips were found"}
		}
		return nil, err
	}
	return &topClipsResponse, nil
}

func (c *client) GetClipCountByBroadcastID(ctx context.Context, broadcastID string, opts *twitchhttp.ReqOpts) (*ClipCount, error) {
	req, err := c.NewRequest("GET", fmt.Sprintf("/api/v2/broadcasts/%s/count", broadcastID), nil)
	if err != nil {
		return nil, err
	}

	clipCount := ClipCount{}
	combinedOpts := twitchhttp.MergeReqOpts(opts, twitchhttp.ReqOpts{
		StatName:       "service.clipsv2.get_broadcast_clip_count",
		StatSampleRate: defaultStatSampleRate,
	})
	_, err = c.DoJSON(ctx, &clipCount, req, combinedOpts)
	if err != nil {
		return nil, err
	}
	return &clipCount, nil
}

func (c *client) GetClipsByBroadcastID(ctx context.Context, params *GetClipsByBroadcastIDParams, opts *twitchhttp.ReqOpts) (*PaginatedClipsResponse, error) {
	if params == nil || params.BroadcastID == "" {
		return nil, errors.New("BroadcastID missing from GetClipsByBroadcastIDParams")
	}

	path := fmt.Sprintf("/api/v2/broadcasts/%s/clips", url.QueryEscape(params.BroadcastID))

	queryParams := url.Values{}
	if params.Limit != nil {
		queryParams.Add("limit", fmt.Sprintf("%d", *params.Limit))
	}
	if params.Sort != nil {
		queryParams.Add("sort", *params.Sort)
	}
	if params.Cursor != nil {
		queryParams.Add("cursor", *params.Cursor)
	}

	if len(queryParams) > 0 {
		path = fmt.Sprintf("%s?%s", path, queryParams.Encode())
	}

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	req.URL.RawQuery = queryParams.Encode()

	response := PaginatedClipsResponse{}
	combinedOpts := twitchhttp.MergeReqOpts(opts, twitchhttp.ReqOpts{
		StatName:       "service.clipsv2.get_clips_by_broadcast_id",
		StatSampleRate: defaultStatSampleRate,
	})
	_, err = c.DoJSON(ctx, &response, req, combinedOpts)
	if err != nil {
		return nil, err
	}
	return &response, nil
}
