package clipsv2

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"time"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/video/clips-upload/client/testutils"
	"code.justin.tv/video/clips-upload/utils/pointers"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"golang.org/x/net/context"
)

var _ = Describe("When making a request with the client", func() {
	Context("Getting a clip by id", func() {
		It("should return a clip and no error on success", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, singleClipResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			createdAt, err := time.Parse(time.RFC3339, "2016-11-01T19:00:16Z")
			Expect(err).To(BeNil())

			clip, err := client.GetClip(context.Background(), "NotLikeThis", NewGetClipParams(), nil)
			Expect(err).To(BeNil())
			Expect(clip).To(Equal(&Clip{
				ID:                     "232232",
				Slug:                   "NotLikeThis",
				URL:                    "https://clips.twitch-test.tv/NotLikeThis",
				EmbedURL:               "https://clips.twitch-test.tv/embed?clip=NotLikeThis",
				EmbedHTML:              "<iframe src='https://clips.twitch-test.tv/embed?clip=NotLikeThis&autoplay=false' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
				BroadcasterID:          "1234",
				BroadcasterDisplayName: "Pikachu",
				BroadcasterLogin:       "pikachu",
				BroadcasterChannelURL:  "https://www.twitch-test.tv/pikachu",
				CuratorID:              "5678",
				CuratorDisplayName:     "MuDkip",
				CuratorLogin:           "mudkip",
				CuratorChannelURL:      "https://www.twitch-test.tv/mudkip",
				VodID:                  pointers.String("1234"),
				VodURL:                 pointers.String("https://www.twitch-test.tv/videos/1234?t=3h28m58s&tt_medium=clips_web&tt_content=full_vod_button"),
				Game:                   "Pokemon",
				Title:                  "Pikachu's stream title",
				Views:                  3154,
				Duration:               30.322,
				CreatedAt:              createdAt,
				Thumbnails: map[string]string{
					"medium": "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-480x272.jpg",
					"small":  "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-260x147.jpg",
					"tiny":   "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-86x45.jpg",
				},
			}))
		})

		It("should return an error when the server errors", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNotFound)
				fmt.Fprintln(w, singleClipErrorResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			_, err = client.GetClip(context.Background(), "NotLikeThisMouse", NewGetClipParams(), nil)
			Expect(err).To(Not(BeNil()))

			Expect(testutils.StatusCode(err)).To(Equal(http.StatusNotFound))
		})

		It("should return a forbidden error when the server returns a forbidden error", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusForbidden)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			_, err = client.GetClip(context.Background(), "NotLikeThisMouse", NewGetClipParams(), nil)
			Expect(err).To(Not(BeNil()))

			Expect(testutils.StatusCode(err)).To(Equal(http.StatusForbidden))
		})
	})

	Context("Getting the clip's status", func() {
		It("should return the clip's status and no error on success", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, clipStatusResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			status, err := client.GetClipStatus(context.Background(), "NotLikeThisMouse", nil)
			Expect(err).To(BeNil())
			Expect(status.Status).To(Equal("created"))
			Expect(len(status.QualityOptions)).To(Equal(2))
		})

		It("should return a forbidden error when the server returns a forbidden error", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusForbidden)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			_, err = client.GetClipStatus(context.Background(), "NotLikeThisMouse", nil)
			Expect(err).To(Not(BeNil()))

			Expect(testutils.StatusCode(err)).To(Equal(http.StatusForbidden))
		})
	})

	Context("Getting the top clips", func() {
		It("should return the clips and no error on success", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, paginatedClipsResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			response, err := client.GetTopClips(context.Background(), NewGetTopClipsParams(), nil)
			Expect(err).To(BeNil())
			Expect(len(response.Clips)).To(Equal(7))
			Expect(response.Clips[0].ID).To(Equal("33292014"))
			Expect(response.Clips[0].Slug).To(Equal("NiceRespectfulVelociraptorSoBayed"))
		})

		It("should return an error when the server errors", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNotFound)
				fmt.Fprintln(w, paginatedClipsErrorResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			_, err = client.GetTopClips(context.Background(), NewGetTopClipsParams(), nil)
			Expect(err).To(Not(BeNil()))
		})
	})

	Context("Getting the broadcast clip count", func() {
		It("should return the broadcast clip count and no error on success", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, getClipCountByBroadcastResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			clipCount, err := client.GetClipCountByBroadcastID(context.Background(), "1234", nil)
			Expect(err).To(BeNil())
			Expect(clipCount.Count).To(Equal(14))
		})

		It("should return an error when the server errors", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(500)
				fmt.Fprintln(w, getClipCountByBroadcastErrorResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			_, err = client.GetClipCountByBroadcastID(context.Background(), "1234", nil)
			Expect(err).To(Not(BeNil()))
		})
	})

	Context("Getting clips by broadcast id", func() {
		It("should return the clips and no error on success", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, paginatedClipsResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			params := &GetClipsByBroadcastIDParams{BroadcastID: "123"}
			response, err := client.GetClipsByBroadcastID(context.Background(), params, nil)
			Expect(err).To(BeNil())
			Expect(len(response.Clips)).To(Equal(7))
			Expect(response.Clips[0].ID).To(Equal("33292014"))
			Expect(response.Clips[0].Slug).To(Equal("NiceRespectfulVelociraptorSoBayed"))
		})

		It("should return an error when the server errors", func() {
			testServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusNotFound)
				fmt.Fprintln(w, paginatedClipsErrorResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: testServer.URL})
			Expect(err).To(BeNil())

			_, err = client.GetTopClips(context.Background(), NewGetTopClipsParams(), nil)
			Expect(err).To(Not(BeNil()))
		})
	})
})

const singleClipErrorResponse = `
{
	"status": "error",
	"message": "no clip data with slug 'NotLikeThisMouse' for broadcaster pikachu exists"
}
`

const singleClipResponse = `
{
	"id": "232232",
	"slug": "NotLikeThis",
	"url": "https://clips.twitch-test.tv/NotLikeThis",
	"embed_url": "https://clips.twitch-test.tv/embed?clip=NotLikeThis",
	"embed_html": "<iframe src='https://clips.twitch-test.tv/embed?clip=NotLikeThis&autoplay=false' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
	"broadcaster_id": "1234",
	"broadcaster_display_name": "Pikachu",
	"broadcaster_login": "pikachu",
	"broadcaster_channel_url": "https://www.twitch-test.tv/pikachu",
	"curator_id": "5678",
	"curator_display_name": "MuDkip",
	"curator_login": "mudkip",
	"curator_channel_url": "https://www.twitch-test.tv/mudkip",
	"vod_id": "1234",
	"vod_url": "https://www.twitch-test.tv/videos/1234?t=3h28m58s&tt_medium=clips_web&tt_content=full_vod_button",
	"game": "Pokemon",
	"title": "Pikachu's stream title",
	"views": 3154,
	"duration": 30.322,
	"created_at": "2016-11-01T19:00:16Z",
	"thumbnails": {
		"medium": "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-480x272.jpg",
		"small": "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-260x147.jpg",
		"tiny": "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-86x45.jpg"
	}
}
`

const clipStatusErrorResponse = `
{
	"status": "error",
	"message": "No such clip"
}
`

const clipStatusResponse = `
{
    "status": "created",
    "quality_options": [
        {
            "quality": "480",
            "source": "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23681131024-offset-30.mp4"
        },
        {
            "quality": "360",
            "source": "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23681131024-offset-30-640x360.mp4"
        }
    ]
}
`

const paginatedClipsErrorResponse = `
{
    "status": "error",
    "message": "No clips found in provided criteria"
}
`

const getClipCountByBroadcastResponse = `
{
    "count": 14
}
`

const getClipCountByBroadcastErrorResponse = `
{
	"status": "error",
	"message": "Internal Server Error"
}
`

const paginatedClipsResponse = `
{
    "clips": [
        {
            "broadcaster_channel_url": "https://www.twitch.tv/eternalenvyy",
            "broadcaster_display_name": "EternaLEnVyy",
            "broadcaster_id": "26954716",
            "broadcaster_login": "eternalenvyy",
            "broadcaster_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/eternalenvyy-profile_image-6b20e441fd6342b1-300x300.png",
            "curator_channel_url": "https://www.twitch.tv/theithovsk",
            "curator_display_name": "theithovsk",
            "curator_id": "55127095",
            "curator_login": "theithovsk",
            "curator_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/theithovsk-profile_image-cdb7d3790a8a0055-300x300.jpeg",
            "preview_image": "https://clips-media-assets.twitch.tv/23021237552-offset-32212-preview.jpg",
            "thumbnails": {
                "medium": "https://clips-media-assets.twitch.tv/23021237552-offset-32212-preview-480x272.jpg",
                "small": "https://clips-media-assets.twitch.tv/23021237552-offset-32212-preview-260x147.jpg",
                "tiny": "https://clips-media-assets.twitch.tv/23021237552-offset-32212-preview-86x45.jpg"
            },
            "game": "Dota 2",
            "created_at": "2016-08-28T11:23:26Z",
            "title": "hey",
            "language": "en",
            "url": "https://clips.twitch.tv/NiceRespectfulVelociraptorSoBayed",
            "status_url": "https://clips.twitch.tv/api/v2/clips/NiceRespectfulVelociraptorSoBayed/status",
            "vod_id": "86109988",
            "vod_url": "https://www.twitch.tv/videos/86109988?t=8h56m52s",
            "embed_url": "https://clips.twitch.tv/embed?clip=NiceRespectfulVelociraptorSoBayed",
            "embed_html": "<iframe src='https://clips.twitch.tv/embed?clip=NiceRespectfulVelociraptorSoBayed' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
            "view_url": "https://clips.twitch.tv/api/v2/clips/NiceRespectfulVelociraptorSoBayed/view",
            "id": "33292014",
            "slug": "NiceRespectfulVelociraptorSoBayed",
            "duration": 30,
            "views": 83963
        },
        {
            "broadcaster_channel_url": "https://www.twitch.tv/eternalenvyy",
            "broadcaster_display_name": "EternaLEnVyy",
            "broadcaster_id": "26954716",
            "broadcaster_login": "eternalenvyy",
            "broadcaster_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/eternalenvyy-profile_image-6b20e441fd6342b1-300x300.png",
            "curator_channel_url": "https://www.twitch.tv/poojedoto",
            "curator_display_name": "poojedoto",
            "curator_id": "51464131",
            "curator_login": "poojedoto",
            "preview_image": "https://clips-media-assets.twitch.tv/23102323584-offset-1178-preview.jpg",
            "thumbnails": {
                "medium": "https://clips-media-assets.twitch.tv/23102323584-offset-1178-preview-480x272.jpg",
                "small": "https://clips-media-assets.twitch.tv/23102323584-offset-1178-preview-260x147.jpg",
                "tiny": "https://clips-media-assets.twitch.tv/23102323584-offset-1178-preview-86x45.jpg"
            },
            "game": "Dota 2",
            "created_at": "2016-09-06T18:54:39Z",
            "title": "Serenity",
            "language": "en",
            "url": "https://clips.twitch.tv/PlayfulCheerfulCourgetteShadyLulu",
            "status_url": "https://clips.twitch.tv/api/v2/clips/PlayfulCheerfulCourgetteShadyLulu/status",
            "vod_id": "87937513",
            "vod_url": "https://www.twitch.tv/videos/87937513?t=19m38s",
            "embed_url": "https://clips.twitch.tv/embed?clip=PlayfulCheerfulCourgetteShadyLulu",
            "embed_html": "<iframe src='https://clips.twitch.tv/embed?clip=PlayfulCheerfulCourgetteShadyLulu' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
            "view_url": "https://clips.twitch.tv/api/v2/clips/PlayfulCheerfulCourgetteShadyLulu/view",
            "id": "31802699",
            "slug": "PlayfulCheerfulCourgetteShadyLulu",
            "duration": 30,
            "views": 73438
        },
        {
            "broadcaster_channel_url": "https://www.twitch.tv/eternalenvyy",
            "broadcaster_display_name": "EternaLEnVyy",
            "broadcaster_id": "26954716",
            "broadcaster_login": "eternalenvyy",
            "broadcaster_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/eternalenvyy-profile_image-6b20e441fd6342b1-300x300.png",
            "curator_channel_url": "https://www.twitch.tv/fledfromnowhere",
            "curator_display_name": "fledfromnowhere",
            "curator_id": "71718706",
            "curator_login": "fledfromnowhere",
            "preview_image": "https://clips-media-assets.twitch.tv/23093248992-offset-4184-preview.jpg",
            "thumbnails": {
                "medium": "https://clips-media-assets.twitch.tv/23093248992-offset-4184-preview-480x272.jpg",
                "small": "https://clips-media-assets.twitch.tv/23093248992-offset-4184-preview-260x147.jpg",
                "tiny": "https://clips-media-assets.twitch.tv/23093248992-offset-4184-preview-86x45.jpg"
            },
            "game": "Dota 2",
            "created_at": "2016-09-05T16:32:16Z",
            "title": "MID or PERU",
            "language": "en",
            "url": "https://clips.twitch.tv/KitschySlipperyAlpacaDancingBaby",
            "status_url": "https://clips.twitch.tv/api/v2/clips/KitschySlipperyAlpacaDancingBaby/status",
            "vod_id": "87729111",
            "vod_url": "https://www.twitch.tv/videos/87729111?t=1h9m44s",
            "embed_url": "https://clips.twitch.tv/embed?clip=KitschySlipperyAlpacaDancingBaby",
            "embed_html": "<iframe src='https://clips.twitch.tv/embed?clip=KitschySlipperyAlpacaDancingBaby' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
            "view_url": "https://clips.twitch.tv/api/v2/clips/KitschySlipperyAlpacaDancingBaby/view",
            "id": "31972972",
            "slug": "KitschySlipperyAlpacaDancingBaby",
            "duration": 29.467,
            "views": 57152
        },
        {
            "broadcaster_channel_url": "https://www.twitch.tv/eternalenvyy",
            "broadcaster_display_name": "EternaLEnVyy",
            "broadcaster_id": "26954716",
            "broadcaster_login": "eternalenvyy",
            "broadcaster_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/eternalenvyy-profile_image-6b20e441fd6342b1-300x300.png",
            "curator_channel_url": "https://www.twitch.tv/m0rgoth_",
            "curator_display_name": "M0rgoth_",
            "curator_id": "67811517",
            "curator_login": "m0rgoth_",
            "curator_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/m0rgoth_-profile_image-c919bf8ac12dbf91-300x300.jpeg",
            "preview_image": "https://clips-media-assets.twitch.tv/23091755136-offset-5380-preview.jpg",
            "thumbnails": {
                "medium": "https://clips-media-assets.twitch.tv/23091755136-offset-5380-preview-480x272.jpg",
                "small": "https://clips-media-assets.twitch.tv/23091755136-offset-5380-preview-260x147.jpg",
                "tiny": "https://clips-media-assets.twitch.tv/23091755136-offset-5380-preview-86x45.jpg"
            },
            "game": "Dota 2",
            "created_at": "2016-09-05T11:26:05Z",
            "title": "MID or PERU",
            "language": "en",
            "url": "https://clips.twitch.tv/VastFragileWoodpecker4Head",
            "status_url": "https://clips.twitch.tv/api/v2/clips/VastFragileWoodpecker4Head/status",
            "vod_id": "87696358",
            "vod_url": "https://www.twitch.tv/videos/87696358?t=1h29m40s",
            "embed_url": "https://clips.twitch.tv/embed?clip=VastFragileWoodpecker4Head",
            "embed_html": "<iframe src='https://clips.twitch.tv/embed?clip=VastFragileWoodpecker4Head' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
            "view_url": "https://clips.twitch.tv/api/v2/clips/VastFragileWoodpecker4Head/view",
            "id": "32003578",
            "slug": "VastFragileWoodpecker4Head",
            "duration": 30,
            "views": 52561
        },
        {
            "broadcaster_channel_url": "https://www.twitch.tv/eternalenvyy",
            "broadcaster_display_name": "EternaLEnVyy",
            "broadcaster_id": "26954716",
            "broadcaster_login": "eternalenvyy",
            "broadcaster_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/eternalenvyy-profile_image-6b20e441fd6342b1-300x300.png",
            "curator_channel_url": "https://www.twitch.tv/theunlimitedbladeworks",
            "curator_display_name": "TheUnlimitedBladeWorks",
            "curator_id": "95508732",
            "curator_login": "theunlimitedbladeworks",
            "preview_image": "https://clips-media-assets.twitch.tv/23469725328-offset-2206-preview.jpg",
            "thumbnails": {
                "medium": "https://clips-media-assets.twitch.tv/23469725328-offset-2206-preview-480x272.jpg",
                "small": "https://clips-media-assets.twitch.tv/23469725328-offset-2206-preview-260x147.jpg",
                "tiny": "https://clips-media-assets.twitch.tv/23469725328-offset-2206-preview-86x45.jpg"
            },
            "game": "Dota 2",
            "created_at": "2016-10-22T02:58:45Z",
            "title": "NP EternaLEnVy",
            "language": "en",
            "url": "https://clips.twitch.tv/SincereTangibleReubenCorgiDerp",
            "status_url": "https://clips.twitch.tv/api/v2/clips/SincereTangibleReubenCorgiDerp/status",
            "vod_id": "96394943",
            "vod_url": "https://www.twitch.tv/videos/96394943?t=36m47s",
            "embed_url": "https://clips.twitch.tv/embed?clip=SincereTangibleReubenCorgiDerp",
            "embed_html": "<iframe src='https://clips.twitch.tv/embed?clip=SincereTangibleReubenCorgiDerp' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
            "view_url": "https://clips.twitch.tv/api/v2/clips/SincereTangibleReubenCorgiDerp/view",
            "id": "24562362",
            "slug": "SincereTangibleReubenCorgiDerp",
            "duration": 29.666999999999998,
            "views": 52282
        },
        {
            "broadcaster_channel_url": "https://www.twitch.tv/eternalenvyy",
            "broadcaster_display_name": "EternaLEnVyy",
            "broadcaster_id": "26954716",
            "broadcaster_login": "eternalenvyy",
            "broadcaster_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/eternalenvyy-profile_image-6b20e441fd6342b1-300x300.png",
            "curator_channel_url": "https://www.twitch.tv/chkomatore",
            "curator_display_name": "chkomatore",
            "curator_id": "81469060",
            "curator_login": "chkomatore",
            "preview_image": "https://clips-media-assets.twitch.tv/23381950848-offset-66-preview.jpg",
            "thumbnails": {
                "medium": "https://clips-media-assets.twitch.tv/23381950848-offset-66-preview-480x272.jpg",
                "small": "https://clips-media-assets.twitch.tv/23381950848-offset-66-preview-260x147.jpg",
                "tiny": "https://clips-media-assets.twitch.tv/23381950848-offset-66-preview-86x45.jpg"
            },
            "game": "Dota 2",
            "created_at": "2016-10-11T03:30:46Z",
            "title": "NP EternaLEnVy",
            "language": "en",
            "url": "https://clips.twitch.tv/HonorableFilthyWalrusKippa",
            "status_url": "https://clips.twitch.tv/api/v2/clips/HonorableFilthyWalrusKippa/status",
            "vod_id": "94239459",
            "vod_url": "https://www.twitch.tv/videos/94239459?t=1m7s",
            "embed_url": "https://clips.twitch.tv/embed?clip=HonorableFilthyWalrusKippa",
            "embed_html": "<iframe src='https://clips.twitch.tv/embed?clip=HonorableFilthyWalrusKippa' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
            "view_url": "https://clips.twitch.tv/api/v2/clips/HonorableFilthyWalrusKippa/view",
            "id": "26400014",
            "slug": "HonorableFilthyWalrusKippa",
            "duration": 30,
            "views": 51100
        },
        {
            "broadcaster_channel_url": "https://www.twitch.tv/eternalenvyy",
            "broadcaster_display_name": "EternaLEnVyy",
            "broadcaster_id": "26954716",
            "broadcaster_login": "eternalenvyy",
            "broadcaster_logo": "https://static-cdn.jtvnw.net/jtv_user_pictures/eternalenvyy-profile_image-6b20e441fd6342b1-300x300.png",
            "curator_channel_url": "https://www.twitch.tv/toastery",
            "curator_display_name": "toastery",
            "curator_id": "105214553",
            "curator_login": "toastery",
            "preview_image": "https://clips-media-assets.twitch.tv/24188538624-offset-4424.313999999968-24-preview.jpg",
            "thumbnails": {
                "medium": "https://clips-media-assets.twitch.tv/24188538624-offset-4424.313999999968-24-preview-480x272.jpg",
                "small": "https://clips-media-assets.twitch.tv/24188538624-offset-4424.313999999968-24-preview-260x147.jpg",
                "tiny": "https://clips-media-assets.twitch.tv/24188538624-offset-4424.313999999968-24-preview-86x45.jpg"
            },
            "game": "Dota 2",
            "created_at": "2017-01-09T16:50:15Z",
            "title": "Stream from Malaysia",
            "language": "en",
            "url": "https://clips.twitch.tv/EnticingDullRutabagaFUNgineer",
            "status_url": "https://clips.twitch.tv/api/v2/clips/EnticingDullRutabagaFUNgineer/status",
            "vod_id": "113523092",
            "vod_url": "https://www.twitch.tv/videos/113523092?t=1h14m11s",
            "embed_url": "https://clips.twitch.tv/embed?clip=EnticingDullRutabagaFUNgineer",
            "embed_html": "<iframe src='https://clips.twitch.tv/embed?clip=EnticingDullRutabagaFUNgineer' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
            "view_url": "https://clips.twitch.tv/api/v2/clips/EnticingDullRutabagaFUNgineer/view",
            "id": "6876728",
            "slug": "EnticingDullRutabagaFUNgineer",
            "duration": 24,
            "views": 49358
        }
    ],
    "cursor": "Nw=="
}
`
