package clips

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
)

type DeleteClipBySlugV3Params struct {
	UserID string `json:"user_id"`
}

func (c *client) DeleteClipBySlugV3(ctx context.Context, userID string, slug string, opts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/api/v3/clips/%s", slug)

	params := DeleteClipBySlugV3Params{
		UserID: userID,
	}
	body, err := json.Marshal(params)
	if err != nil {
		return err
	}

	req, err := c.NewRequest("DELETE", path, bytes.NewBuffer(body))
	if err != nil {
		return err
	}

	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.delete_clip_by_slug_v3",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedOpts)
	if err != nil {
		return err
	}

	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}
	return nil
}

type BatchDeleteClipsV3Params struct {
	UserID string   `json:"user_id"`
	Slugs  []string `json:"slugs"`
}

func (c *client) BatchDeleteClipsV3(ctx context.Context, userID string, slugs []string, opts *twitchclient.ReqOpts) (DeletedClips, error) {
	path := "/api/v3/clips"

	params := BatchDeleteClipsV3Params{
		UserID: userID,
		Slugs:  slugs,
	}
	body, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("DELETE", path, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}

	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.batch_delete_clips_v3",
		StatSampleRate: defaultStatSampleRate,
	})

	deletedClips := DeletedClips{}
	resp, err := c.DoJSON(ctx, &deletedClips, req, combinedOpts)
	if err != nil {
		if resp != nil && resp.StatusCode == http.StatusNotFound {
			return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "no clips were found"}
		}
		return nil, err
	}
	return deletedClips, nil
}

type DeleteClipByVODV3Params struct {
	UserID string `json:"user_id"`
}

func (c *client) DeleteClipsByVODV3(ctx context.Context, userID, vodID string, opts *twitchclient.ReqOpts) (DeletedClips, error) {
	path := fmt.Sprintf("/api/v3/vods/%s/clips", vodID)

	params := DeleteClipByVODV3Params{
		UserID: userID,
	}
	body, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("DELETE", path, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}

	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.delete_clip_by_vod_v3",
		StatSampleRate: defaultStatSampleRate,
	})

	deletedClips := DeletedClips{}
	_, err = c.DoJSON(ctx, &deletedClips, req, combinedOpts)
	if err != nil {
		return nil, err
	}
	return deletedClips, nil
}

type DeleteClipByBroadcastV3Params struct {
	UserID string `json:"user_id"`
}

func (c *client) DeleteClipsByBroadcastV3(ctx context.Context, userID, broadcastID string, opts *twitchclient.ReqOpts) (DeletedClips, error) {
	path := fmt.Sprintf("/api/v3/broadcasts/%s/clips", broadcastID)

	params := DeleteClipByBroadcastV3Params{
		UserID: userID,
	}
	body, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("DELETE", path, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}

	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.delete_clip_by_broadcast_v3",
		StatSampleRate: defaultStatSampleRate,
	})

	deletedClips := DeletedClips{}
	_, err = c.DoJSON(ctx, &deletedClips, req, combinedOpts)
	if err != nil {
		return nil, err
	}
	return deletedClips, nil
}

type ReportClipBySlugV3Params struct {
	UserID      string `json:"user_id"`
	Reason      string `json:"reason"`
	Description string `json:"description"`
}

func (c *client) ReportClipBySlugV3(ctx context.Context, userID, slug, reason, description string, opts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/api/v3/clips/%s/report", slug)

	params := ReportClipBySlugV3Params{
		UserID:      userID,
		Reason:      reason,
		Description: description,
	}
	body, err := json.Marshal(params)
	if err != nil {
		return err
	}

	req, err := c.NewRequest("POST", path, bytes.NewBuffer(body))
	if err != nil {
		return err
	}

	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.report_clip_by_slug_v3",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedOpts)
	if err != nil {
		return err
	}

	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}
	return nil
}

type BanUserV3Params struct {
	UserID string `json:"user_id"`
}

func (c *client) BanUserV3(ctx context.Context, channelID, requesterUserID, userToBanID string, isTemporary bool, opts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/api/v3/channels/%s/users/%s/ban", channelID, userToBanID)
	if isTemporary {
		path = fmt.Sprintf("%s?temporary=true", path)
	}

	params := BanUserV3Params{
		UserID: requesterUserID,
	}
	body, err := json.Marshal(params)
	if err != nil {
		return err
	}

	req, err := c.NewRequest("POST", path, bytes.NewBuffer(body))
	if err != nil {
		return err
	}

	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.ban_user_v3",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.Do(ctx, req, combinedOpts)
	if err != nil {
		return err
	}

	if resp.StatusCode >= 400 {
		return twitchclient.HandleFailedResponse(resp)
	}
	return nil
}

type EditClipTitleV3Params struct {
	UserID string `json:"user_id"`
	Title  string `json:"title"`
}

func (c *client) EditClipTitleV3(ctx context.Context, userID, slug, title string, opts *twitchclient.ReqOpts) (*Clip, error) {
	path := fmt.Sprintf("/api/v3/clips/%s/title", slug)

	params := EditClipTitleV3Params{
		UserID: userID,
		Title:  title,
	}

	body, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("POST", path, bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}

	clip := Clip{}
	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.edit_clip_title_v3",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.DoJSON(ctx, &clip, req, combinedOpts)
	if err != nil {
		if resp != nil && resp.StatusCode == http.StatusNotFound {
			return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "clip not found"}
		}
		return nil, err
	}
	return &clip, nil
}

func (c *client) GetClipInfoV3(ctx context.Context, slug string, params *GetClipInfoParams, opts *twitchclient.ReqOpts) (*Clip, error) {
	if params == nil {
		params = NewGetClipInfoParams()
	}

	req, err := c.NewRequest("GET", fmt.Sprintf("/api/v3/clips/%s?%s", slug, params.URLParamsString()), nil)
	if err != nil {
		return nil, err
	}

	clip := Clip{}
	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.get_clip_info_v3",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.DoJSON(ctx, &clip, req, combinedOpts)
	if err != nil {
		if resp != nil {
			return nil, constructSingleClipTwitchError(resp)
		}
		return nil, err
	}
	return &clip, nil
}

func (c *client) GetTopClipsV3(ctx context.Context, params *GetTopClipsParams, opts *twitchclient.ReqOpts) (*PaginatedClipsResponse, error) {
	if params == nil {
		params = NewGetTopClipsParams()
	}

	req, err := c.NewRequest("GET", fmt.Sprintf("/api/v3/clips/top?%s", params.URLParamsString()), nil)
	if err != nil {
		return nil, err
	}

	topClipsResponse := PaginatedClipsResponse{}
	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.get_top_clips_v3",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.DoJSON(ctx, &topClipsResponse, req, combinedOpts)
	if err != nil {
		if resp != nil && resp.StatusCode == http.StatusNotFound {
			return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "no clips were found"}
		}
		return nil, err
	}
	return &topClipsResponse, nil
}

func (c *client) GetMyClipsV3(ctx context.Context, params *GetMyClipsParams, opts *twitchclient.ReqOpts) (*PaginatedClipsResponse, error) {
	if params == nil {
		params = NewGetMyClipsParams()
	}

	req, err := c.NewRequest("GET", fmt.Sprintf("/api/v3/clips/me?%s", params.URLParamsString()), nil)
	if err != nil {
		return nil, err
	}

	myClipsResponse := PaginatedClipsResponse{}
	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.clips.get_my_clips_v3",
		StatSampleRate: defaultStatSampleRate,
	})
	resp, err := c.DoJSON(ctx, &myClipsResponse, req, combinedOpts)
	if err != nil {
		if resp != nil && resp.StatusCode == http.StatusNotFound {
			return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "no clips were found"}
		}
		return nil, err
	}
	return &myClipsResponse, nil
}

func constructSingleClipTwitchError(resp *http.Response) error {
	if resp.StatusCode == http.StatusForbidden {
		return &twitchclient.Error{StatusCode: http.StatusForbidden, Message: "this channel has been closed due to terms of service violation"}
	} else if resp.StatusCode == http.StatusNotFound {
		return &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "clip not found"}
	}

	return twitchclient.HandleFailedResponse(resp)
}
