package clips

import (
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/video/clips-upload/utils/pointers"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"golang.org/x/net/context"
)

var _ = Describe("When making a request to API V3 with the client", func() {
	var (
		ctx          context.Context
		userID       string
		slug         string
		expectedClip *Clip
	)

	BeforeEach(func() {
		ctx = context.Background()
		userID = "12345"
		slug = "thisisareallyreallylongslug"
		createdAt, err := time.Parse(time.RFC3339, "2016-11-01T19:00:16Z")
		Expect(err).To(BeNil())

		expectedClip = &Clip{
			ID:                     "232232",
			Slug:                   "NotLikeThis",
			URL:                    "https://clips.twitch-test.tv/NotLikeThis",
			EmbedURL:               "https://clips.twitch-test.tv/embed?clip=NotLikeThis",
			EmbedHTML:              "<iframe src='https://clips.twitch-test.tv/embed?clip=NotLikeThis&autoplay=false' width='640' height='360' frameborder='0' scrolling='no' allowfullscreen='true'></iframe>",
			BroadcasterID:          "1234",
			BroadcasterDisplayName: "Pikachu",
			BroadcasterLogin:       "pikachu",
			BroadcasterChannelURL:  "https://www.twitch-test.tv/pikachu",
			CuratorID:              "5678",
			CuratorDisplayName:     "MuDkip",
			CuratorLogin:           "mudkip",
			CuratorChannelURL:      "https://www.twitch-test.tv/mudkip",
			VodID:                  pointers.String("1234"),
			VodURL:                 pointers.String("https://www.twitch-test.tv/videos/1234?t=3h28m58s&tt_medium=clips_web&tt_content=full_vod_button"),
			Game:                   "Pokemon",
			Title:                  "Pikachu's stream title",
			Views:                  3154,
			Duration:               30.322,
			CreatedAt:              createdAt,
			Thumbnails: map[string]string{
				"medium": "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-480x272.jpg",
				"small":  "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-260x147.jpg",
				"tiny":   "https://s3-us-west-2.amazonaws.com/twitch-clips-staging/23553545680-offset-12538-preview-86x45.jpg",
			},
		}
	})

	Context("Deleting the clip by slug", func() {
		It("should return no error on success", func() {
			expectedRequestBody := map[string]interface{}{
				"user_id": userID,
			}

			testServer := initTestServer(fmt.Sprintf("/api/v3/clips/%s", slug), nil, expectedRequestBody)
			defer testServer.Close()

			client := initClient(testServer.URL)

			err := client.DeleteClipBySlugV3(ctx, userID, slug, nil)
			Expect(err).To(BeNil())
		})

		It("should return an error on failure", func() {
			testServer := initBrokenTestServer(http.StatusInternalServerError)
			defer testServer.Close()

			client := initClient(testServer.URL)

			err := client.DeleteClipBySlugV3(ctx, userID, slug, nil)
			Expect(err).To(Not(BeNil()))
		})
	})

	Context("Getting the clip info", func() {
		It("should return the clip info with no error on success", func() {
			testServer := initTestServer(fmt.Sprintf("/api/v3/clips/%s?with_url_tracking=%t", slug, false), expectedClip, nil)
			defer testServer.Close()

			client := initClient(testServer.URL)

			clip, err := client.GetClipInfoV3(ctx, slug, NewGetClipInfoParams(), nil)
			Expect(err).To(BeNil())
			Expect(clip).To(Equal(expectedClip))
		})

		It("should return an error on failure", func() {
			testServer := initBrokenTestServer(http.StatusNotFound)
			defer testServer.Close()

			client := initClient(testServer.URL)

			_, err := client.GetClipInfoV3(ctx, slug, NewGetClipInfoParams(), nil)
			Expect(err).To(Not(BeNil()))
		})
	})

	Describe("GetTopClipsV3", func() {
		It("should return the clips and no error on success", func() {
			expectedTopClips := &PaginatedClipsResponse{
				Clips:  []*Clip{expectedClip, expectedClip},
				Cursor: "Nw==",
			}
			params := NewGetTopClipsParams()
			testServer := initTestServer(fmt.Sprintf("/api/v3/clips/top?%s", params.URLParamsString()), expectedTopClips, nil)
			defer testServer.Close()

			client := initClient(testServer.URL)
			topClips, err := client.GetTopClipsV3(ctx, params, nil)

			Expect(err).To(BeNil())
			Expect(topClips).To(Equal(expectedTopClips))
		})

		Context("when the server errors", func() {
			It("should return an error when the server errors", func() {
				params := NewGetTopClipsParams()
				testServer := initBrokenTestServer(http.StatusNotFound)
				defer testServer.Close()

				client := initClient(testServer.URL)
				_, err := client.GetTopClipsV3(ctx, params, nil)

				Expect(err).To(Not(BeNil()))
			})
		})
	})

	Describe("GetMyClipsV3", func() {
		It("should return the clips and no error on success", func() {
			expectedMyClips := &PaginatedClipsResponse{
				Clips:  []*Clip{expectedClip, expectedClip},
				Cursor: "Nw==",
			}
			params := NewGetMyClipsParams()
			testServer := initTestServer(fmt.Sprintf("/api/v3/clips/me?%s", params.URLParamsString()), expectedMyClips, nil)
			defer testServer.Close()

			client := initClient(testServer.URL)
			myClips, err := client.GetMyClipsV3(ctx, params, nil)

			Expect(err).To(BeNil())
			Expect(myClips).To(Equal(expectedMyClips))
		})

		Context("when the server errors", func() {
			It("should return an error when the server errors", func() {
				params := NewGetMyClipsParams()
				testServer := initBrokenTestServer(http.StatusNotFound)
				defer testServer.Close()

				client := initClient(testServer.URL)
				_, err := client.GetMyClipsV3(ctx, params, nil)

				Expect(err).To(Not(BeNil()))
			})
		})
	})

	Describe("BatchDeleteClipsV3", func() {
		It("should return no error on success", func() {
			slugs := []string{expectedClip.Slug}
			expectedRequestBody := map[string]interface{}{
				"user_id": userID,
				"slugs":   []interface{}{expectedClip.Slug},
			}

			expectedDeletedClips := DeletedClips{
				&DeletedClip{
					ID:   expectedClip.ID,
					Slug: expectedClip.Slug,
				},
			}

			testServer := initTestServer("/api/v3/clips", expectedDeletedClips, expectedRequestBody)
			defer testServer.Close()

			client := initClient(testServer.URL)
			deletedClips, err := client.BatchDeleteClipsV3(ctx, userID, slugs, nil)
			Expect(err).To(BeNil())
			Expect(deletedClips).To(Equal(expectedDeletedClips))
		})

		It("should return an error when the server errors", func() {
			slugs := []string{expectedClip.Slug}
			testServer := initBrokenTestServer(http.StatusInternalServerError)
			defer testServer.Close()

			client := initClient(testServer.URL)
			deletedClips, err := client.BatchDeleteClipsV3(ctx, userID, slugs, nil)
			Expect(err).To(Not(BeNil()))
			Expect(deletedClips).To(BeNil())
		})
	})

	Describe("DeleteClipsByVODV3", func() {
		const vodID = "123420"

		It("should return no error on success", func() {
			expectedRequestBody := map[string]interface{}{
				"user_id": userID,
			}

			expectedDeletedClips := DeletedClips{
				&DeletedClip{
					ID:   expectedClip.ID,
					Slug: expectedClip.Slug,
				},
			}

			testServer := initTestServer(fmt.Sprintf("/api/v3/vods/%s/clips", vodID), expectedDeletedClips, expectedRequestBody)
			defer testServer.Close()

			client := initClient(testServer.URL)
			deletedClips, err := client.DeleteClipsByVODV3(ctx, userID, vodID, nil)
			Expect(err).To(BeNil())
			Expect(deletedClips).To(Equal(expectedDeletedClips))
		})

		It("should return an error when the server errors", func() {
			testServer := initBrokenTestServer(http.StatusInternalServerError)
			defer testServer.Close()

			client := initClient(testServer.URL)
			deletedClips, err := client.DeleteClipsByVODV3(ctx, userID, vodID, nil)
			Expect(err).To(Not(BeNil()))
			Expect(deletedClips).To(BeNil())
		})
	})

	Describe("DeleteClipsByBroadcastV3", func() {
		const broadcastID = "123420"

		It("should return no error on success", func() {
			expectedRequestBody := map[string]interface{}{
				"user_id": userID,
			}

			expectedDeletedClips := DeletedClips{
				&DeletedClip{
					ID:   expectedClip.ID,
					Slug: expectedClip.Slug,
				},
			}

			testServer := initTestServer(fmt.Sprintf("/api/v3/broadcasts/%s/clips", broadcastID), expectedDeletedClips, expectedRequestBody)
			defer testServer.Close()

			client := initClient(testServer.URL)
			deletedClips, err := client.DeleteClipsByBroadcastV3(ctx, userID, broadcastID, nil)
			Expect(err).To(BeNil())
			Expect(deletedClips).To(Equal(expectedDeletedClips))
		})

		It("should return an error when the server errors", func() {
			testServer := initBrokenTestServer(http.StatusInternalServerError)
			defer testServer.Close()

			client := initClient(testServer.URL)
			deletedClips, err := client.DeleteClipsByBroadcastV3(ctx, userID, broadcastID, nil)
			Expect(err).To(Not(BeNil()))
			Expect(deletedClips).To(BeNil())
		})
	})

	Describe("ReportClipBySlugV3", func() {
		const (
			reason      = "no reason"
			description = "no description"
		)

		It("should return no error on success", func() {
			expectedRequestBody := map[string]interface{}{
				"user_id":     userID,
				"reason":      reason,
				"description": description,
			}

			testServer := initTestServer(fmt.Sprintf("/api/v3/clips/%s/report", slug), nil, expectedRequestBody)
			defer testServer.Close()

			client := initClient(testServer.URL)
			err := client.ReportClipBySlugV3(ctx, userID, slug, reason, description, nil)
			Expect(err).To(BeNil())
		})

		It("should return an error when the server errors", func() {
			testServer := initBrokenTestServer(http.StatusInternalServerError)
			defer testServer.Close()

			client := initClient(testServer.URL)
			err := client.ReportClipBySlugV3(ctx, userID, slug, reason, description, nil)
			Expect(err).To(Not(BeNil()))
		})
	})

	Describe("DeleteClipsByBroadcastV3", func() {
		var (
			channelID       = userID
			requesterUserID = userID
			userToBanID     = "300"
			isTemporary     = false
		)

		It("should return no error on success", func() {
			expectedRequestBody := map[string]interface{}{
				"user_id": requesterUserID,
			}

			path := fmt.Sprintf("/api/v3/channels/%s/users/%s/ban", channelID, userToBanID)
			testServer := initTestServer(path, nil, expectedRequestBody)
			defer testServer.Close()

			client := initClient(testServer.URL)
			err := client.BanUserV3(ctx, channelID, requesterUserID, userToBanID, isTemporary, nil)
			Expect(err).To(BeNil())
		})

		It("should return an error when the server errors", func() {
			testServer := initBrokenTestServer(http.StatusInternalServerError)
			defer testServer.Close()

			client := initClient(testServer.URL)
			err := client.BanUserV3(ctx, channelID, requesterUserID, userToBanID, isTemporary, nil)
			Expect(err).To(Not(BeNil()))
		})
	})

	Describe("EditClipTitleV3", func() {
		const (
			title = "New Title"
		)

		It("should return no error on success", func() {
			expectedRequestBody := map[string]interface{}{
				"user_id": userID,
				"title":   title,
			}

			testServer := initTestServer(fmt.Sprintf("/api/v3/clips/%s/title", slug), expectedClip, expectedRequestBody)
			defer testServer.Close()

			client := initClient(testServer.URL)
			updatedClipInfo, err := client.EditClipTitleV3(ctx, userID, slug, title, nil)
			Expect(err).To(BeNil())
			Expect(updatedClipInfo).To(Equal(expectedClip))
		})

		It("should return an error when the server errors", func() {
			testServer := initBrokenTestServer(http.StatusInternalServerError)
			defer testServer.Close()

			client := initClient(testServer.URL)
			updatedClipInfo, err := client.EditClipTitleV3(ctx, userID, slug, title, nil)
			Expect(err).To(Not(BeNil()))
			Expect(updatedClipInfo).To(BeNil())
		})
	})
})
