multiJob('video-clips-multijob') {
    concurrentBuild true

    triggers {
        githubPush()
    }

    configure { project ->
        project / builders / 'com.cloudbees.jenkins.GitHubSetCommitStatusBuilder'
    }

    wrappers {
        sshAgent 'git-aws-read-key'
        preBuildCleanup()
        timestamps()
        credentialsBinding {
            string('GITHUB_ACCESS_TOKEN', 'devtools-deployment-github-api-token')
        }
    }

    publishers {
        githubCommitNotifier()
        flexiblePublish {
            conditionalAction {
                condition {
                    and {
                        // only trigger against master
                        stringsMatch('$GIT_BRANCH', 'origin/master', false)
                    } {
                        status('SUCCESS', 'SUCCESS')
                    }
                }
                steps {
                    shell """
                        set -x
                        echo Auto-Deploy
                        export AUTO_DEPLOY_ENV=production
                        export BRANCH=\$(echo "\$GIT_BRANCH" | sed 's/origin\\///')
                        ./deploy/auto-deploy.sh -b "\$BRANCH" -e "\$AUTO_DEPLOY_ENV" -c "\$GIT_COMMIT" -t "\$GITHUB_ACCESS_TOKEN"
                    """
                }
            }
        }
    }

    scm {
        git {
            remote {
                github 'video/clips-upload', 'ssh', 'git-aws.internal.justin.tv'
                credentials 'git-aws-read-key'
            }
            clean true
        }
    }

    steps {
        phase('Build Clips', 'SUCCESSFUL') {
            job('video-clips-server', true) {
                gitRevision(true)
                killPhaseCondition('NEVER')
            }
            job('video-clips-client', true) {
                gitRevision(true)
                killPhaseCondition('NEVER')
            }
        }
    }
}

job {
    name 'video-clips-TEMPLATE'
    concurrentBuild true

    scm {
        git {
            remote {
                github 'video/clips-upload', 'ssh', 'git-aws.internal.justin.tv'
                credentials 'git-aws-read-key'
            }
            clean true
        }
    }

    wrappers {
        sshAgent 'git-aws-read-key'
        preBuildCleanup()
        timestamps()
        credentialsBinding {
            file('COURIERD_PRIVATE_KEY', 'courierd')
            file('AWS_CONFIG_FILE', 'aws_config')

            string 'AWS_ACCESS_KEY', 'video-aws-access-key'
            string 'AWS_SECRET_KEY', 'video-aws-secret-key'
        }
    }
}

job {
    name 'video-clips-server'
    using 'video-clips-TEMPLATE'

    steps {
        shell './deploy/build.sh api'
        shell './deploy/push.sh api'
        saveDeployArtifact 'video/clips-api', 'current_deploy/'

        shell './deploy/build.sh worker'
        shell './deploy/push.sh worker'
        saveDeployArtifact 'video/clips-worker', 'current_deploy/'
    }
}

job {
    name 'video-clips-client'
    using 'video-clips-TEMPLATE'

    steps {
        shell 'manta -v -f ./manta/assets.json'
        saveDeployArtifact 'video/clips-client', '.manta/assets/'
    }
}

multiJob('video-clips-deploy-job') {
    parameters {
        booleanParam('TRIGGER_SMOCA', false)
        stringParam('GITHUB_CREATOR', '')
        stringParam 'ENVIRONMENT'
        stringParam 'GIT_COMMIT'
    }

    wrappers {
        credentialsBinding {
            file('AWS_CONFIG_FILE', 'aws_config')

            string 'AWS_ACCESS_KEY', 'video-aws-access-key'
            string 'AWS_SECRET_KEY', 'video-aws-secret-key'
        }
    }

  steps {
    phase('Deploy') {
      phaseJob('video-clips-deploy') {
        killPhaseCondition('FAILURE')
      }
      phaseJob('video-clips-internal-deploy') {
        killPhaseCondition('FAILURE')
      }
    }
  }
}

freeStyleJob('video-clips-internal-deploy') {
    using 'TEMPLATE-deploy-aws'

     parameters {
        booleanParam('TRIGGER_SMOCA', false)
        stringParam('GITHUB_CREATOR', '')
    }

    wrappers {
        credentialsBinding {
            file('AWS_CONFIG_FILE', 'aws_config')

            string 'AWS_ACCESS_KEY', 'video-aws-access-key'
            string 'AWS_SECRET_KEY', 'video-aws-secret-key'
        }
    }

    steps {
        downloadDeployArtifact 'video/clips-api'
        shell """
                |set +x
                |export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
                |export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
                |set -x
                |
                |if [ "\$ENVIRONMENT" = "production" ]; then
                |  eb deploy --timeout 60 clips-production-internal | tee /dev/stderr | grep -q "update completed successfully"
                |elif [ "\$ENVIRONMENT" = "beta" ]; then
                |  eb deploy --timeout 60 beta-internal | tee /dev/stderr | grep -q "update completed successfully"
                |fi
            """.stripMargin()

        shell 'rm -rf * .ebextensions .elasticbeanstalk'
    }
}


freeStyleJob('video-clips-deploy') {
    using 'TEMPLATE-deploy-aws'

    parameters {
        booleanParam('TRIGGER_SMOCA', false)
        stringParam('GITHUB_CREATOR', '')
    }

    wrappers {
        credentialsBinding {
            file('AWS_CONFIG_FILE', 'aws_config')

            string 'AWS_ACCESS_KEY', 'video-aws-access-key'
            string 'AWS_SECRET_KEY', 'video-aws-secret-key'
        }
    }

    steps {
        // deploy frontend assets
        downloadDeployArtifact 'video/clips-client'
        shell """pip install awsebcli >/dev/null
                |set +x
                |export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
                |export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
                |set -x
                |
                |aws s3 cp . s3://twitch-clips-assets/\$ENVIRONMENT/assets/\$GIT_COMMIT --recursive --acl public-read --metadata-directive REPLACE --cache-control public,max-age=3153600
                |if [ "\$ENVIRONMENT" = "production" ]; then
                |  aws s3 cp . s3://twitch-clips-assets/production-external/assets/\$GIT_COMMIT --recursive --acl public-read --metadata-directive REPLACE --cache-control public,max-age=3153600
                |fi
              """.stripMargin()
        shell 'rm -rf *'


        // TODO: removed the aws cli install line, not sure if this will still work
        downloadDeployArtifact 'video/clips-api'
        shell """
                |set +x
                |export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
                |export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
                |set -x
                |
                |if [ "\$ENVIRONMENT" = "production" ]; then
                |  eb deploy --timeout 60 clips-production-external | tee /dev/stderr | grep -q "update completed successfully"
                |elif [ "\$ENVIRONMENT" = "beta" ]; then
                |  eb deploy --timeout 60 clips-beta | tee /dev/stderr | grep -q "update completed successfully"
                |else
                |  eb deploy --timeout 60 clips-\$ENVIRONMENT | tee /dev/stderr | grep -q "update completed successfully"
                |fi
              """.stripMargin()

        shell 'rm -rf * .ebextensions .elasticbeanstalk'

        downloadDeployArtifact 'video/clips-worker'
        shell """
                |set +x
                |export AWS_ACCESS_KEY_ID=\$AWS_ACCESS_KEY
                |export AWS_SECRET_ACCESS_KEY=\$AWS_SECRET_KEY
                |set -x
                |
                |if [ "\$ENVIRONMENT" = "staging" ]; then
                |  eb deploy clips-worker-staging | tee /dev/stderr | grep -q "update completed successfully"
                |elif [ "\$ENVIRONMENT" = "production" ]; then
                |  eb deploy clips-worker-production | tee /dev/stderr | grep -q "update completed successfully"
                |fi
              """.stripMargin()

        shell 'rm -rf * .ebextensions .elasticbeanstalk'

        // Run Smoca downstream if TRIGGER_SMOCA is true
        conditionalSteps {
          condition {
            booleanCondition('$TRIGGER_SMOCA')
          }
          runner('Fail')
          steps {
            downstreamParameterized {
              trigger('smoca-clips', 'ALWAYS') {
                parameters {
                  currentBuild()
                }
              }
            }
          }
        }
    }
}
