package main

import (
	// core

	"fmt"
	"log"
	"net/url"
	"strings"
	"time"

	// app-specific

	"code.justin.tv/video/clips-upload/adapters/db"
	"code.justin.tv/video/clips-upload/adapters/hystrixclients"
	"code.justin.tv/video/clips-upload/adapters/network"
	"code.justin.tv/video/clips-upload/adapters/repositories"
	"code.justin.tv/video/clips-upload/api"
	. "code.justin.tv/video/clips-upload/api/log"
	"code.justin.tv/video/clips-upload/app"
	"code.justin.tv/video/clips-upload/app/video"
	"code.justin.tv/video/clips-upload/clients"
	"code.justin.tv/video/clips-upload/clients/follows"
	"code.justin.tv/video/clips-upload/clients/mako"
	"code.justin.tv/video/clips-upload/clients/recommendations"
	"code.justin.tv/video/clips-upload/clients/spade"
	"code.justin.tv/video/clips-upload/datastore"
	i18n "code.justin.tv/video/clips-upload/i18n/client"
	"code.justin.tv/video/clips-upload/utils/lstats"
	"code.justin.tv/video/clips-upload/utils/slugs"
	"code.justin.tv/video/clips-upload/utils/stats"
	vinylClient "code.justin.tv/vod/vinyl/client"

	// vendor
	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/video/clips-upload/adapters/reporters"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sts"

	// video transcoding
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/elastictranscoder"

	appconfig "code.justin.tv/video/clips-upload/app/config"
)

// this should get over-written by using the -X linker flag
var serviceVersion = "development"

func throwUndefinedVariablePanic(varName string) {
	panic(fmt.Sprintf("environment variable %s not specified", varName))
}

func init() {
	config.Register(map[string]string{
		"ENVIRONMENT":                   "",
		"AWS_ACCESS_KEY":                "",
		"AWS_SECRET_KEY":                "",
		"ASSETS_ROOT_URL":               "",
		"ASSETS_PROXY_URL":              "",
		"TWITCH_API_URL":                "",
		"TWITCH_API_INTERNAL_URL":       "",
		"USERS_SERVICE_URL":             "",
		"MONEYPENNY_URL":                "",
		"FEEDS_EDGE_URL":                "",
		"FOLLOWING_SERVICE_URL":         "",
		"MAKO_URL":                      "",
		"CLIPS_ORIGIN_URL":              "",
		"CLIPS_INFO_TABLE":              "",
		"CLIPS_BLACKLIST_TABLE":         "",
		"VIDEO_BUCKET":                  "",
		"RAW_MEDIA_BUCKET":              "",
		"VIDEO_ROOT_URL":                "",
		"RAW_VIDEO_ROOT_URL":            "",
		"OWL_URL":                       "",
		"REDIS_URL":                     "",
		"SPADE_HOST_URL":                "",
		"PG_DB_HOSTNAME":                "",
		"PG_DB_PORT":                    "",
		"PG_DB_NAME":                    "",
		"PG_DB_USER":                    "",
		"PG_DB_PASS":                    "",
		"PGBOUNCER_DB_HOSTNAME":         "",
		"PGBOUNCER_DB_NAME":             "",
		"PG_SLAVE_HOSTNAME":             "",
		"PG_SLAVE_PORT":                 "",
		"PG_SLAVE_NAME":                 "",
		"PG_SLAVE_USER":                 "",
		"PG_SLAVE_PASS":                 "",
		"RECOMMENDATIONS_URL":           "",
		"LEVIATHAN_AUTHORIZATION_TOKEN": "",
		"LEVIATHAN_URL":                 "",
		"JAX_URL":                       "",
		"AET_INPUT_BUCKET":              "",
		"AET_PIPELINE_ID":               "",
		"AET_PRESET_IDS":                "",
		"CLIPS_ROLE_ARN":                "",
		"NOTIFICATIONS_SNS_TOPIC_ARN":   "",
		"MAKO_SNS_TOPIC_ARN":            "",
		"TMI_HOSTPORT":                  "",
		"VINYL_URL":                     "",
		"ZUMA_URL":                      "",
	})
}

func main() {
	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	env := config.MustResolve("ENVIRONMENT")

	region := "us-west-2"
	awsAccessKey := config.MustResolve("AWS_ACCESS_KEY")
	awsSecretKey := config.MustResolve("AWS_SECRET_KEY")

	assetsRootURL := config.MustResolve("ASSETS_ROOT_URL")
	assetsProxyURL := config.Resolve("ASSETS_PROXY_URL")

	twitchAPIURL := config.MustResolve("TWITCH_API_URL")
	usersServiceURL := config.MustResolve("USERS_SERVICE_URL")
	moneypennyURL := config.MustResolve("MONEYPENNY_URL")
	feedsEdgeURL := config.MustResolve("FEEDS_EDGE_URL")
	recommendationsURL := config.MustResolve("RECOMMENDATIONS_URL")
	followingServiceURL := config.MustResolve("FOLLOWING_SERVICE_URL")
	makoURL := config.MustResolve("MAKO_URL")

	clipsOriginURL := config.MustResolve("CLIPS_ORIGIN_URL")

	videoBucket := config.MustResolve("VIDEO_BUCKET")
	rawClipMediaBucket := config.MustResolve("RAW_MEDIA_BUCKET")

	videoRootURL := config.MustResolve("VIDEO_ROOT_URL")
	rawVideoRootURL := config.MustResolve("RAW_VIDEO_ROOT_URL")

	owlURL := config.MustResolve("OWL_URL")

	redisURL := config.MustResolve("REDIS_URL")

	spadeHostURL := config.MustResolve("SPADE_HOST_URL")

	pgDBHost := config.MustResolve("PG_DB_HOSTNAME")
	pgDBPort := config.MustResolve("PG_DB_PORT")
	pgDBName := config.MustResolve("PG_DB_NAME")
	pgDBUser := config.MustResolve("PG_DB_USER")
	pgDBPass := config.MustResolve("PG_DB_PASS")

	pgSlaveHost := config.MustResolve("PG_SLAVE_HOSTNAME")
	pgSlavePort := config.MustResolve("PG_SLAVE_PORT")
	pgSlaveName := config.MustResolve("PG_SLAVE_NAME")
	pgSlaveUser := config.MustResolve("PG_SLAVE_USER")
	pgSlavePass := config.MustResolve("PG_SLAVE_PASS")

	pgbouncerDBHost := config.MustResolve("PGBOUNCER_DB_HOSTNAME")
	pgbouncerDBName := config.MustResolve("PGBOUNCER_DB_NAME")

	leviathanAuthToken := config.MustResolve("LEVIATHAN_AUTHORIZATION_TOKEN")
	leviathanURL := config.MustResolve("LEVIATHAN_URL")
	jaxURL := config.MustResolve("JAX_URL")

	aetInputBucket := config.MustResolve("AET_INPUT_BUCKET")
	aetPipelineID := config.MustResolve("AET_PIPELINE_ID")
	aetPresetIDs := config.MustResolve("AET_PRESET_IDS")
	clipsRoleARN := config.MustResolve("CLIPS_ROLE_ARN")
	notificiationsSNSTopicARN := config.MustResolve("NOTIFICATIONS_SNS_TOPIC_ARN")
	makoSNSTopicARN := config.MustResolve("MAKO_SNS_TOPIC_ARN")
	tmiHostPort := config.MustResolve("TMI_HOSTPORT")
	vinylURL := config.MustResolve("VINYL_URL")
	zumaURL := config.MustResolve("ZUMA_URL")

	assetsURLWithServiceVerison := fmt.Sprintf("%s/%s", assetsRootURL, serviceVersion)

	if env == "development" && assetsProxyURL != "" && assetsRootURL != assetsProxyURL {
		assetsRootURL = assetsProxyURL
		assetsURLWithServiceVerison = assetsProxyURL
	}

	parsedTwitchAPIURL, err := url.Parse(twitchAPIURL)
	if err != nil {
		panic("twitch API URL could not be parsed")
	}

	_, err = url.Parse(rawVideoRootURL)
	if err != nil {
		panic("raw video root URL could not be parsed")
	}

	logger := NewMultiLogger(NewLogrusLogger(), NewRollbarLogger())

	dataRepository := repositories.NewS3DataRepository(
		region,
		videoBucket,
		rawClipMediaBucket,
		aetInputBucket,
		s3.BucketCannedACLPublicRead,
		awsAccessKey,
		awsSecretKey)

	statsClient, err := stats.NewStats(env)
	if err != nil {
		log.Fatal(err)
	}

	logStats, err := lstats.NewClient(statsClient, logger)
	if err != nil {
		logger.Error(err)
	}

	defer func() {
		err := statsClient.Close()
		if err != nil {
			log.Print(err.Error())
		}
	}()

	spadeClient, err := spade.NewClient(spadeHostURL, logStats, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create spade event reporter %s", err.Error()))
		return
	}

	// twitchhttp expects the host to always be set, however - we don't know the host name
	// until right before we issue the playlist request.  We supply the full domain and path
	// when issuing the request
	playlistHTTPClient, err := network.NewNetworkHttpClient("temp", "video", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	twitchAPIHTTPClient, err := network.NewNetworkHttpClient(twitchAPIURL, "twitch_api", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	moneypennyClient, err := clients.NewMoneypennyClient(moneypennyURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create moneypenny service client %s", err.Error()))
		return
	}

	feedsClient, err := clients.NewFeedsClient(feedsEdgeURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create feeds client %s", err.Error()))
	}

	recommendationsClient, err := recommendations.NewClient(recommendationsURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create recommendations client %s", err.Error()))
	}

	followsClient := follows.NewClient(followingServiceURL, statsClient)

	makoClient := mako.NewClient(makoURL, statsClient)

	usersServiceClient, err := clients.NewUsersServiceClient(usersServiceURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create users service client %s", err.Error()))
		return
	}

	channelsClient, err := clients.NewChannelsClient(usersServiceURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create channels client %s", err.Error()))
		return
	}

	userPropertiesHTTPClient, err := network.NewNetworkHttpClient("https://api.internal.twitch.tv", "user_properties", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	usherHTTPClient, err := network.NewNetworkHttpClient("http://usher.ttvnw.net", "usher", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	leviathanHTTPClient, err := network.NewNetworkHttpClient(leviathanURL, "leviathan", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	clipCreatorHTTPClient, err := network.NewNetworkHttpClient(assetsRootURL, "clip_creator", statsClient, logger)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	jaxClient, err := clients.NewJaxClient(jaxURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create http client %s", err.Error()))
		return
	}

	zumaClient, err := clients.NewZumaClient(zumaURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create zuma client %s", err.Error()))
		return
	}

	twitchRepository := repositories.NewTwitchRepository(twitchAPIHTTPClient, parsedTwitchAPIURL, logStats, logger)

	mediaRepository := repositories.NewUsherMediaRepository(playlistHTTPClient, video.HLSPlaylistParser, logStats, logger)

	channelRepository := repositories.NewTwitchChannelRepository(twitchAPIHTTPClient, statsClient, logger)
	vodRepository := repositories.NewTwitchVODRepository(twitchAPIHTTPClient, statsClient, logger)

	owlClient, err := clients.NewOwlClient(owlURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create owl client from owl/client library %s", err.Error()))
		return
	}
	hystrixclients.InitHystrix(logger)
	owlHystrixClient := hystrixclients.NewOwlClient(owlClient)

	oauthTokenRepository := repositories.NewOauthTokenRepository(twitchAPIHTTPClient, twitchAPIURL, owlHystrixClient, statsClient, logger)
	userPropertiesRepository := repositories.NewUserPropertiesRepository(userPropertiesHTTPClient, logger)

	usherPlaylistRepository := repositories.NewUsherPlaylistRepository(usherHTTPClient, statsClient, logger)

	redisClient, err := db.NewRedisClient(redisURL, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to redis client: %s", err.Error()))
		return
	}

	pgConn, err := db.NewPostgresConn(pgDBHost, pgDBPort, pgDBName, pgDBUser, pgDBPass, logStats)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create connection to database: %s", err.Error()))
		return
	}
	go db.CollectMetrics(pgConn, statsClient)

	pgSlaveConn, err := db.NewPostgresConn(pgSlaveHost, pgSlavePort, pgSlaveName, pgSlaveUser, pgSlavePass, logStats)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create connection to slave: %s", err.Error()))
		return
	}
	go db.CollectMetrics(pgSlaveConn, statsClient)

	pgbouncerConn, err := db.NewPostgresConn(pgbouncerDBHost, "6432", pgbouncerDBName, pgDBUser, pgDBPass, logStats)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create connection to pgbouncer %s", err.Error()))
		return
	}

	viewCountsRepository := repositories.NewViewCountsRepository(pgConn, redisClient, statsClient, logger)
	acknowledgementRepository := repositories.NewAcknowledgementRepository(pgConn, statsClient, logger)

	slugGenerator := slugs.NewGenerator(logStats)

	clipsDatastore := datastore.NewDatastore(pgConn, pgSlaveConn, pgbouncerConn, redisClient, logStats, logger)

	awsCredentials := credentials.NewStaticCredentials(awsAccessKey, awsSecretKey, "")
	awsSession := session.New(&aws.Config{
		Region:      aws.String(region),
		Credentials: awsCredentials,
	})
	aetClient := elastictranscoder.New(awsSession)

	clipCreator := video.NewClipCreator(
		mediaRepository,
		dataRepository,
		clipsDatastore,
		video.NewCommandRunner(logStats, logger),
		logStats,
		logger,
		rawVideoRootURL,
		clipCreatorHTTPClient,
		aetClient,
		aetPipelineID,
		strings.Split(aetPresetIDs, ","),
	)

	leviathanClipReporter := reporters.NewLeviathanClipReporter(leviathanHTTPClient, leviathanAuthToken, leviathanURL, logger)
	cacheInvalidator := network.NewCloudwatchCCNCacheInvalidator(region, awsAccessKey, awsSecretKey)

	languageMatcher := i18n.GetSupportedLanguageMatcher()
	clientI18N := i18n.NewClientI18N(languageMatcher)

	snsClient := configureSNSClient(clipsRoleARN)

	appConfig := appconfig.Config{
		NotificationSNSTopicARN: notificiationsSNSTopicARN,
		MakoSNSTopicARN:         makoSNSTopicARN,
	}

	tmiClient, err := clients.NewTMIClient(tmiHostPort, statsClient)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create tmi client %s", err))
		return
	}

	vinylConf := twitchclient.ClientConf{
		Host: vinylURL,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 200,
		},
		Stats: statsClient,
	}

	newVinylClient, err := vinylClient.NewClient(vinylConf)
	if err != nil {
		logger.Error(fmt.Errorf("Failed to create vinyl client %s", err.Error()))
		return
	}

	clients := &clients.Clients{
		I18N:                 clientI18N,
		OauthTokenRepository: oauthTokenRepository,
		TwitchRepository:     twitchRepository,
		UsersServiceClient:   usersServiceClient,
		ChannelsClient:       channelsClient,
		VinylClient:          newVinylClient,
		ZumaClient:           zumaClient,
		MoneypennyClient:     moneypennyClient,
	}

	app := app.NewApp(
		clipsDatastore,
		clients,
		dataRepository,
		slugGenerator,
		channelRepository,
		userPropertiesRepository,
		feedsClient,
		jaxClient,
		recommendationsClient,
		followsClient,
		makoClient,
		clipCreator,
		leviathanClipReporter,
		usherPlaylistRepository,
		mediaRepository,
		vodRepository,
		viewCountsRepository,
		acknowledgementRepository,
		oauthTokenRepository,
		cacheInvalidator,
		spadeClient,
		videoRootURL,
		redisClient,
		snsClient,
		tmiClient,
		appConfig,
		logStats,
		logger,
	)

	// server
	server := api.NewServer(
		clients,
		redisClient,
		spadeClient,
		assetsURLWithServiceVerison,
		clipsOriginURL,
		twitchAPIURL,
		videoRootURL,
		serviceVersion,
		logStats,
		logger,
		app,
	)

	// By default this handles SIGUSR2, SIGINT, SIGTERM OS signals.
	// https://git-aws.internal.justin.tv/common/twitchhttp/blob/282c1a72b779c0cf35c3a9d406cb02af03a29e18/server.go#L110
	twitchhttp.AddDefaultSignalHandlers()
	serverConfig := twitchhttp.NewConfig()
	// 30 seconds is the longest it takes for a clip to be created / published.
	serverConfig.GracefulTimeout = 30 * time.Second
	log.Fatal(twitchhttp.ListenAndServe(server, serverConfig))
}

func configureSNSClient(role string) *sns.SNS {
	stsclient := sts.New(session.New(&aws.Config{
		Region: aws.String("us-west-2"),
	}))

	arp := &stscreds.AssumeRoleProvider{
		ExpiryWindow: 1 * time.Second,
		RoleARN:      role,
		Client:       stsclient,
	}

	creds := credentials.NewCredentials(arp)

	session := session.New(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	return sns.New(session)
}
