const webpack = require('webpack');
const path = require('path');
const CleanWebpackPlugin = require('clean-webpack-plugin');
const ExtractTextPlugin = require('extract-text-webpack-plugin');
var Dashboard = require('webpack-dashboard');
var DashboardPlugin = require('webpack-dashboard/plugin');

const BASE_DIR = './assets';
const CODE_DIR = `${BASE_DIR}/lib`;
const GENERATED_CODE_DIR = `${BASE_DIR}/generated`;
const TEST_DIR = `${BASE_DIR}/spec`;
const ENV_DEVELOPMENT = 'development';
const ENV_PRODUCTION = 'production';
const ENV_TEST = 'test';
const IS_PRODUCTION = process.env.NODE_ENV === ENV_PRODUCTION;
const IS_TEST = process.env.NODE_ENV === ENV_TEST;
const OUTPUT_DIR = `${BASE_DIR}/${IS_PRODUCTION ? 'dist' : 'development'}`;
const SOURCE_MAP = sourceMap();
const DASHBOARD = process.env.DASHBOARD;

// Set the VERSION for tracking:
// Use env var GIT_COMMIT exposed by manta, locally use git to look up revision
if (process.env.GIT_COMMIT) {
  process.env.VERSION = process.env.GIT_COMMIT;
} else {
  const git = require('git-rev-sync');
  process.env.VERSION = git.long();
}

/**
* NOTE: Ensure you mirror any plugins you set here with those in Gruntfile.js
*/
const BABEL_OPTIONS = {
  presets: [ 'es2015', 'react' ],
  plugins: [
    'babel-polyfill',
    'transform-async-functions',
    'transform-regenerator',
    'transform-class-properties',
    'transform-object-rest-spread'
  ]
};

if (IS_PRODUCTION) {
    // deopts include using 'ref' or object rest spread in jsx
  BABEL_OPTIONS.plugins.concat([
    'transform-react-inline-elements',
    'transform-react-constant-elements'
  ]);
}

const DEFAULT_PLUGINS = [
  new CleanWebpackPlugin([ './assets/development', './assets/dist' ], { verbose: false }),
  new ExtractTextPlugin('[name].css'),
  new webpack.EnvironmentPlugin([ 'NODE_ENV', 'VERSION' ])
];

const productionPlugins = [
  new webpack.optimize.UglifyJsPlugin({
    compress: {
      warnings: true
    }
  })
];

const developmentPlugins = DASHBOARD ? [ new DashboardPlugin(new Dashboard().setData) ] : [];

/**
* @param {string} environment
* @return {Array}
*/
const PLUGINS = IS_PRODUCTION ?
DEFAULT_PLUGINS.concat(productionPlugins) :
DEFAULT_PLUGINS.concat(developmentPlugins);

let EXTERNALS = {
  'react/addons': true,
  'react/lib/ExecutionEnvironment': true,
  'react/lib/ReactContext': true,
};

if (IS_TEST) {
  EXTERNALS = Object.assign({}, EXTERNALS, {
    'react-test-renderer/shallow': true,
    'react-dom/test-utils': true
  });
}

module.exports = {
  entry: {
    'app': [ 'babel-polyfill', `./${CODE_DIR}/app.js` ],
    'embed/player': [ 'babel-polyfill', `${CODE_DIR}/player/index.js` ],
    'view': [ 'babel-polyfill', `./${CODE_DIR}/view.js` ],
    'edit': [ 'babel-polyfill', `./${CODE_DIR}/edit.js` ],
    'error': [ 'babel-polyfill', `./${CODE_DIR}/error.js` ]
  },
  module: {
    rules: [
      {
        test: /\.(js|jsx)$/,
        include: path.resolve(__dirname, CODE_DIR),
        exclude: /(node_modules)/,
        use: {
          loader: 'babel-loader',
          options: BABEL_OPTIONS
        }
      },
      {
        test: /\.(js|jsx)$/,
        include: path.resolve(__dirname, GENERATED_CODE_DIR),
        use: {
          loader: 'babel-loader',
          options: BABEL_OPTIONS
        }
      },
      {
        test: /\.sass$/i,
        use: ExtractTextPlugin.extract([
          {
            loader: 'css-loader',
            options: {
              minimize: IS_PRODUCTION,
              sourceMap: !IS_PRODUCTION
            }
          },
          {
            loader: 'postcss-loader',
            options: {
              plugins: function () {
                return [
                  require('autoprefixer')
                ];
              }
            }
          },
          {
            loader: 'sass-loader?sourceMap?indentedSyntax=true',
            options: {
              data: "@import 'core.variables.sass'",
              includePaths: [
                path.resolve(__dirname, 'node_modules/twitch-core-ui/src/sass'),
                path.resolve(__dirname, 'assets'),
              ],
            },
          }
        ])
      },
      {
        test: /\.(js|jsx)$/,
        enforce: 'pre',
        include: path.resolve(__dirname, TEST_DIR),
        exclude: /(node_modules)/,
        use: {
          loader: 'babel-loader',
          options: BABEL_OPTIONS
        }
      },
    ]
  },
  output: {
    filename: '[name].js',
    library: 'Twitch',
    libraryTarget: 'umd',
    path: path.resolve(__dirname, OUTPUT_DIR)
  },
  resolve: {
    modules: [
      path.resolve('./assets'),
      'node_modules'
    ],
    alias: {
      'jquery-ui/slider': 'jquery-ui/ui/widgets/slider.js'
    }
  },
  bail: true,
  cache: true,
  devtool: IS_PRODUCTION ? false : SOURCE_MAP,
  plugins: PLUGINS,
  externals: EXTERNALS
};

function sourceMap() {
  switch(process.env.NODE_ENV) {
  case ENV_TEST:
    return 'inline-source-map';
  case ENV_DEVELOPMENT:
    return 'source-map';
  default:
    return null;
  }
}
