package cohesion

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/cohesion/api/responses/v1"
	"code.justin.tv/web/cohesion/associations"

	"github.com/jixwanwang/apiutils"

	// We should load the possible schemas for the client
	_ "code.justin.tv/web/cohesion/schema"
	"github.com/cactus/go-statsd-client/statsd"
)

// ErrorResponse describes an error coming from Cohesion
type ErrorResponse struct {
	Status     int    `json:"status"`
	Message    string `json:"message"`
	StatusText string `json:"error"`
}

// Client represents a cohesion client
type Client struct {
	host           string
	httpClientFunc func(ctx context.Context) (*http.Client, error)
	repo           string
	schemaManager  *associations.Schema
}

const (
	timeout = 500 * time.Millisecond
)

var emptyBody = map[string]interface{}{}
var statNameKey string

// DefaultHTTPClient returns a function which can be passed into NewClient to set a custom client per request. Can just pass in a *http.Client.
// Adds in statsd metrics by default.
func DefaultHTTPClient(stats statsd.Statter, statsPrefix string) func(*Client) {
	return func(client *Client) {
		client.httpClientFunc = func(ctx context.Context) (*http.Client, error) {
			return statsHTTPClient(ctx, stats, statsPrefix)
		}
	}
}

// CustomHTTPClient accepts a func(context.Context) *http.Client if you want your http client to be a function of the context.
// Statsd integration is not provided with the custom httpClient.
func CustomHTTPClient(httpClient func(context.Context) (*http.Client, error)) func(*Client) {
	return func(client *Client) {
		client.httpClientFunc = httpClient
	}
}

type statsTransport struct {
	ctx context.Context
	http.Transport
	stats       statsd.Statter
	statsPrefix string
}

type goodRoundTripper interface {
	http.RoundTripper
	CancelRequest(*http.Request)
	CloseIdleConnections()
}

var _ goodRoundTripper = (*statsTransport)(nil)

func statsHTTPClient(ctx context.Context, stats statsd.Statter, statsPrefix string) (*http.Client, error) {
	st := &statsTransport{
		Transport:   http.Transport{},
		ctx:         ctx,
		stats:       stats,
		statsPrefix: statsPrefix,
	}
	rt, err := chitin.RoundTripper(ctx, st)
	if err != nil {
		return nil, err
	}
	return &http.Client{
		Timeout:   timeout,
		Transport: rt,
	}, nil
}

func (st *statsTransport) RoundTrip(req *http.Request) (*http.Response, error) {
	reqStart := time.Now()
	resp, err := st.Transport.RoundTrip(req)
	if err != nil {
		return nil, err
	}
	dur := time.Since(reqStart)
	statName := st.ctx.Value(statNameKey)
	_ = st.stats.TimingDuration(fmt.Sprintf("%s.%s.%d", st.statsPrefix, statName, resp.StatusCode), dur, 0.1)
	return resp, err
}

// SetRepository tells cohesion which project is making calls to the service
func SetRepository(repo string) func(*Client) {
	return func(client *Client) {
		client.repo = fmt.Sprintf("code.justin.tv/%v", repo)
	}
}

// SortAsc can be passed into the ListAssoc method to set the sort param
func SortAsc(req *Request) {
	req.Query.Set("sort", "asc")
}

// SortDesc can be passed into the ListAssoc method to set the sort param
func SortDesc(req *Request) {
	req.Query.Set("sort", "desc")
}

// Offset returns a function which can be passed into the ListAssoc method to set the offset param
func Offset(offset int) func(*Request) {
	return func(req *Request) {
		req.Query.Set("offset", strconv.Itoa(offset))
	}
}

// Limit returns a function which can be passed into the ListAssoc method to set the limit param
func Limit(limit int) func(*Request) {
	return func(req *Request) {
		req.Query.Set("limit", strconv.Itoa(limit))
	}
}

// Cursor returns a function which can be passed into the ListAssoc method to set the cursor param
func Cursor(cursor string) func(*Request) {
	return func(req *Request) {
		req.Query.Set("cursor", cursor)
	}
}

// Data returns a function which can be passed into CreateAssoc to set association metadata
func Data(data map[string]interface{}) func(*Request) {
	return func(req *Request) {
		for k, v := range data {
			req.Body[k] = v
		}
	}
}

// NewClient returns a new cohesion client
// Host can be host or host:port.
// Requires a http Client function that is the result of either DefaultHTTPClient or CustomHTTPClient.
func NewClient(host string, httpClienfFunc func(*Client), schema string, opts ...func(*Client)) (*Client, error) {
	if host == "" {
		return nil, errors.New("cohesion: invalid host")
	}

	host = strings.TrimPrefix(host, "http://")

	c := Client{host: host}
	httpClienfFunc(&c)

	for _, opt := range opts {
		opt(&c)
	}

	schemaManager, err := associations.NewSchemaManager(schema)
	if err != nil {
		return nil, err
	}
	c.schemaManager = schemaManager

	return &c, nil
}

// CreateAssoc creates associations. Pass in Data() to set association metadata.
func (c *Client) CreateAssoc(ctx context.Context, e1 Entity, kind string, e2 Entity, databag map[string]interface{}, opts ...func(*Request)) error {
	if err := c.schemaManager.Validate(e1.Kind, kind, e2.Kind); err != nil {
		return err
	}

	ctx = context.WithValue(ctx, statNameKey, "create_assoc")
	req := newRequest("PUT", fmt.Sprintf("/v1/associations/%s/%s/%s/%s/%s", e1.Kind, e1.ID, kind, e2.Kind, e2.ID), databag)
	var err error
	req.HTTPClient, err = c.httpClientFunc(ctx)
	if err != nil {
		return err
	}

	for _, opt := range opts {
		opt(req)
	}

	httpResp, err := c.do(req, nil)
	if err != nil {
		return err
	}
	switch httpResp.StatusCode {
	case http.StatusCreated:
		return nil
	case apiutils.StatusUnprocessableEntity:
		return associations.ErrAssociationExists{}
	default:
		return fmt.Errorf("cohesion: unexpected status %d", httpResp.StatusCode)
	}
}

// DeleteAssoc deletes associations.
func (c *Client) DeleteAssoc(ctx context.Context, e1 Entity, kind string, e2 Entity, opts ...func(*Request)) error {
	if err := c.schemaManager.Validate(e1.Kind, kind, e2.Kind); err != nil {
		return err
	}

	ctx = context.WithValue(ctx, statNameKey, "delete_assoc")
	req := newRequest("DELETE", fmt.Sprintf("/v1/associations/%s/%s/%s/%s/%s", e1.Kind, e1.ID, kind, e2.Kind, e2.ID), emptyBody)
	var err error
	req.HTTPClient, err = c.httpClientFunc(ctx)
	if err != nil {
		return err
	}

	for _, opt := range opts {
		opt(req)
	}

	httpResp, err := c.do(req, nil)
	if err != nil {
		return err
	}

	switch httpResp.StatusCode {
	case http.StatusNoContent:
		return nil
	case http.StatusNotFound:
		return associations.ErrNotFound{}
	default:
		return fmt.Errorf("cohesion: unexpected status %d", httpResp.StatusCode)
	}
}

// BulkDeleteAssoc deletes all associations of a type
func (c *Client) BulkDeleteAssoc(ctx context.Context, e1 Entity, kind string, toKind string, opts ...func(*Request)) error {
	if err := c.schemaManager.Validate(e1.Kind, kind, toKind); err != nil {
		return err
	}

	ctx = context.WithValue(ctx, statNameKey, "bulk delete_assoc")
	req := newRequest("DELETE", fmt.Sprintf("/v1/associations/%s/%s/%s/%s", e1.Kind, e1.ID, kind, toKind), emptyBody)
	var err error
	req.HTTPClient, err = c.httpClientFunc(ctx)
	if err != nil {
		return err
	}

	for _, opt := range opts {
		opt(req)
	}

	httpResp, err := c.do(req, nil)
	if err != nil {
		return err
	}

	switch httpResp.StatusCode {
	case http.StatusNoContent:
		return nil
	case http.StatusNotFound:
		return associations.ErrNotFound{}
	default:
		return fmt.Errorf("cohesion: unexpected status %d", httpResp.StatusCode)
	}
}

// UpdateAssoc updates the databag of a single association, or changes its association type
func (c *Client) UpdateAssoc(ctx context.Context, e1 Entity, kind string, e2 Entity, newAssocKind string, databag map[string]interface{}, opts ...func(*Request)) error {
	if err := c.schemaManager.Validate(e1.Kind, kind, e2.Kind); err != nil {
		return err
	}

	requestBody := map[string]interface{}{
		"data_bag": databag,
	}
	if newAssocKind != "" {
		if err := c.schemaManager.Validate(e1.Kind, newAssocKind, e2.Kind); err != nil {
			return err
		}
		requestBody["new_assoc_kind"] = newAssocKind
	}

	ctx = context.WithValue(ctx, statNameKey, "update_assoc")
	req := newRequest("POST", fmt.Sprintf("/v1/associations/%s/%s/%s/%s/%s", e1.Kind, e1.ID, kind, e2.Kind, e2.ID), requestBody)
	var err error
	req.HTTPClient, err = c.httpClientFunc(ctx)
	if err != nil {
		return err
	}

	for _, opt := range opts {
		opt(req)
	}

	httpResp, err := c.do(req, nil)
	if err != nil {
		return err
	}

	switch httpResp.StatusCode {
	case http.StatusNoContent:
		return nil
	case http.StatusNotFound:
		return associations.ErrNotFound{}
	default:
		return fmt.Errorf("cohesion: unexpected status %d", httpResp.StatusCode)
	}
}

// FetchAssoc fetches one association.
func (c *Client) FetchAssoc(ctx context.Context, e1 Entity, kind string, e2 Entity, opts ...func(*Request)) (*v1.Response, error) {
	if err := c.schemaManager.Validate(e1.Kind, kind, e2.Kind); err != nil {
		return nil, err
	}

	ctx = context.WithValue(ctx, statNameKey, "fetch_assoc")
	req := newRequest("GET", fmt.Sprintf("/v1/associations/%s/%s/%s/%s/%s", e1.Kind, e1.ID, kind, e2.Kind, e2.ID), emptyBody)
	var err error
	req.HTTPClient, err = c.httpClientFunc(ctx)
	if err != nil {
		return nil, err
	}

	for _, opt := range opts {
		opt(req)
	}

	resp := &v1.Response{}

	httpResp, err := c.do(req, resp)
	if err != nil {
		return nil, err
	}

	switch httpResp.StatusCode {
	case http.StatusNotFound:
		return nil, nil
	case http.StatusOK:
		return resp, nil
	default:
		return nil, fmt.Errorf("cohesion: unexpected status %d", httpResp.StatusCode)
	}
}

// ListAssoc fetches all associations of a certain kind for a user. Pass in SortAsc, SortDesc, Offset(0), Limit(10) options.
func (c *Client) ListAssoc(ctx context.Context, e1 Entity, kind string, toKind string, opts ...func(*Request)) (*v1.Response, error) {
	if err := c.schemaManager.Validate(e1.Kind, kind, toKind); err != nil {
		return nil, err
	}

	ctx = context.WithValue(ctx, statNameKey, "list_assoc")
	req := newRequest("GET", fmt.Sprintf("/v1/associations/%s/%s/%s/%s", e1.Kind, e1.ID, kind, toKind), emptyBody)
	var err error
	req.HTTPClient, err = c.httpClientFunc(ctx)
	if err != nil {
		return nil, err
	}
	req.Query.Set("sort", "asc")
	req.Query.Set("offset", "0")
	req.Query.Set("limit", "100")

	for _, opt := range opts {
		opt(req)
	}

	resp := &v1.Response{}

	httpResp, err := c.do(req, resp)
	if err != nil {
		return nil, err
	}

	if httpResp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("cohesion: unexpected status %d", httpResp.StatusCode)
	}

	return resp, nil
}

// CountAssoc returns the number of associations of a certain kind for a user.
func (c *Client) CountAssoc(ctx context.Context, e1 Entity, kind string, toKind string, opts ...func(*Request)) (int, error) {
	if err := c.schemaManager.Validate(e1.Kind, kind, toKind); err != nil {
		return 0, err
	}

	ctx = context.WithValue(ctx, statNameKey, "count_assoc")
	req := newRequest("GET", fmt.Sprintf("/v1/associations/%s/%s/%s/%s/count", e1.Kind, e1.ID, kind, toKind), emptyBody)
	var err error
	req.HTTPClient, err = c.httpClientFunc(ctx)
	if err != nil {
		return 0, err
	}

	for _, opt := range opts {
		opt(req)
	}

	resp := map[string]int{}

	httpResp, err := c.do(req, &resp)
	if err != nil {
		return 0, err
	}

	if httpResp.StatusCode != http.StatusOK {
		return 0, fmt.Errorf("cohesion: unexpected status %d", httpResp.StatusCode)
	}

	return resp["count"], nil
}

func (c *Client) do(req *Request, resp interface{}) (*http.Response, error) {
	req.Host = c.host

	httpReq, err := req.httpRequest()
	if err != nil {
		return nil, err
	}

	if c.repo != "" {
		httpReq.Header.Add("Twitch-Repository", c.repo)
	}

	httpResp, err := req.HTTPClient.Do(httpReq)
	if err != nil {
		return httpResp, err
	}
	defer func() {
		_ = httpResp.Body.Close()
	}()

	responseBytes, err := ioutil.ReadAll(httpResp.Body)
	if err != nil {
		return httpResp, err
	}

	if httpResp.StatusCode >= 500 {
		errResponse := &ErrorResponse{}
		err = json.Unmarshal(responseBytes, errResponse)
		if err != nil {
			return httpResp, fmt.Errorf("Could not parse error response: %s", err.Error())
		}
		return httpResp, fmt.Errorf("Received status code: %d, %s", httpResp.StatusCode, errResponse.Message)
	}

	if resp != nil {
		err = json.Unmarshal(responseBytes, resp)
		if err != nil {
			return httpResp, err
		}
	}

	return httpResp, nil
}

// Request is used by option functions to decorate http requests
type Request struct {
	Body       map[string]interface{}
	Host       string
	Method     string
	Path       string
	Query      url.Values
	HTTPClient *http.Client
}

func newRequest(method string, path string, body map[string]interface{}) *Request {
	return &Request{
		Body:       body,
		Method:     method,
		Path:       path,
		Query:      url.Values{},
		HTTPClient: http.DefaultClient,
	}
}

func (r *Request) httpRequest() (*http.Request, error) {
	if !strings.HasPrefix(r.Path, "/") {
		return nil, errors.New("cohesion: path must begin with slash")
	}

	url := url.URL{
		Scheme:   "http",
		Host:     r.Host,
		Path:     r.Path,
		RawQuery: r.Query.Encode(),
	}

	switch r.Method {
	case "GET", "DELETE":
		return http.NewRequest(r.Method, url.String(), nil)
	case "POST", "PUT", "PATCH":
		if r.Body == nil {
			r.Body = emptyBody
		}

		bodyBytes, err := json.Marshal(r.Body)
		if err != nil {
			return nil, err
		}

		req, err := http.NewRequest(r.Method, url.String(), bytes.NewReader(bodyBytes))
		if err != nil {
			return nil, err
		}
		req.Header.Add("Content-Type", "application/json")

		return req, nil
	default:
		return nil, errors.New("cohesion: unrecognized http method")
	}
}
