package v2

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/web/cohesion/rpc"
)

func (c *v2Client) newRequest(from Entity, kind string, to Entity, data map[string]interface{}) (*rpc.RequestV2, error) {
	assoc, err := newAssociation(from, kind, to, data, "")
	if err != nil {
		return nil, err
	}

	return &rpc.RequestV2{
		Source:       c.source,
		Options:      new(rpc.RequestOptions),
		Associations: []*rpc.Association{assoc},
	}, nil
}

func (c *v2Client) newBulkRequest(from Entity, kind string, bulkKind string, data map[string]interface{}) (*rpc.RequestV2, error) {
	assoc, err := newAssociation(from, kind, Entity{}, data, bulkKind)
	if err != nil {
		return nil, err
	}

	return &rpc.RequestV2{
		Source:       c.source,
		Options:      new(rpc.RequestOptions),
		Associations: []*rpc.Association{assoc},
	}, nil
}

func (c *v2Client) newBatchGetRequest(from Entity, kind string, bulkKind string, toIDs []string) (*rpc.RequestV2, error) {
	assocs := []*rpc.Association{}

	for _, id := range toIDs {
		to := Entity{
			ID:   id,
			Kind: bulkKind,
		}
		assoc, err := newAssociation(from, kind, to, map[string]interface{}{}, "")
		if err != nil {
			return nil, err
		}

		assocs = append(assocs, assoc)
	}

	return &rpc.RequestV2{
		Source:       c.source,
		Options:      new(rpc.RequestOptions),
		Associations: assocs,
	}, nil
}

func (c *v2Client) newBatchRequest(from Entity, kind string, bulkKind string, data map[string]interface{}) (*rpc.RequestV2, error) {
	assocs := []*rpc.Association{}

	for id, bag := range data {
		if blob, ok := bag.(map[string]interface{}); ok && len(blob) == 0 {
			return nil, fmt.Errorf("error: %s %s data bag is empty", bulkKind, id)
		} else if ok {
			to := Entity{
				ID:   id,
				Kind: bulkKind,
			}
			assoc, err := newAssociation(from, kind, to, blob, "")
			if err != nil {
				return nil, err
			}

			assocs = append(assocs, assoc)
		}
	}

	return &rpc.RequestV2{
		Source:       c.source,
		Options:      new(rpc.RequestOptions),
		Associations: assocs,
	}, nil
}

func newAssociation(from Entity, kind string, to Entity, data map[string]interface{}, bulkKind string) (*rpc.Association, error) {
	bytes, err := json.Marshal(data)
	if err != nil {
		return nil, err
	}

	return &rpc.Association{
		Kind:     kind,
		From:     newEntity(from),
		To:       newEntity(to),
		Data:     bytes,
		BulkKind: bulkKind,
	}, nil
}

func newEntity(e Entity) *rpc.Entity {
	if e.Empty() {
		return nil
	}

	return &rpc.Entity{
		Kind: e.Kind,
		Name: e.ID,
	}
}
