package v2

import (
	"encoding/json"
	"testing"

	"code.justin.tv/web/cohesion/rpc"

	. "github.com/smartystreets/goconvey/convey"
)

func TestNewRequest(t *testing.T) {
	Convey("When creating a new request", t, func() {
		var req *rpc.RequestV2
		var err error

		client := &v2Client{source: "foo/bar"}
		from := Entity{ID: "123", Kind: "user"}
		kind := "follow"
		to := Entity{ID: "456", Kind: "user"}
		data := map[string]interface{}{}

		Convey("It should create a single association", func() {
			req, err = client.newRequest(from, kind, to, data)

			So(len(req.Associations), ShouldEqual, 1)
			So(err, ShouldBeNil)
		})
	})
}

func TestNewBulkRequest(t *testing.T) {
	Convey("When creating a new bulk request", t, func() {
		var req *rpc.RequestV2
		var err error

		client := &v2Client{source: "foo/bar"}
		from := Entity{ID: "123", Kind: "user"}
		kind := "follow"
		data := map[string]interface{}{}
		bulkKind := "user"

		Convey("It should create an association with no To Entity", func() {
			req, err = client.newBulkRequest(from, kind, bulkKind, data)

			So(len(req.Associations), ShouldEqual, 1)
			So(req.Associations[0].To, ShouldBeNil)
			So(err, ShouldBeNil)
		})
	})
}

func TestNewBatchRequest(t *testing.T) {
	Convey("When creating a new batch request", t, func() {
		var req *rpc.RequestV2
		var err error

		client := &v2Client{source: "foo/bar"}
		from := Entity{ID: "123", Kind: "user"}
		kind := "follow"
		data := map[string]interface{}{}
		bulkKind := "user"

		Convey("It should return 0 associations if data is empty", func() {
			req, err = client.newBatchRequest(from, kind, bulkKind, data)

			So(len(req.Associations), ShouldEqual, 0)
			So(err, ShouldBeNil)
		})

		Convey("It should return the correct number of associations", func() {
			data = map[string]interface{}{
				"456": map[string]interface{}{
					"data_bag": map[string]interface{}{
						"block_notifications": false,
					},
				},
				"789": map[string]interface{}{
					"data_bag": map[string]interface{}{
						"block_notifications": true,
					},
					"new_assoc_kind": "foo",
				},
			}
			req, err = client.newBatchRequest(from, kind, bulkKind, data)

			So(len(req.Associations), ShouldEqual, 2)
			So(err, ShouldBeNil)
		})

		Convey("It should return an error for an empty data bag", func() {
			data = map[string]interface{}{
				"456": map[string]interface{}{
					"data_bag": map[string]interface{}{
						"block_notifications": false,
					},
				},
				"789": map[string]interface{}{},
			}
			req, err = client.newBatchRequest(from, kind, bulkKind, data)

			So(req, ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "data bag is empty")
		})
	})
}

func TestNewAssociation(t *testing.T) {
	Convey("When creating a new RPC Association", t, func() {
		var assoc *rpc.Association
		var err error

		from := Entity{ID: "123", Kind: "user"}
		kind := "follow"
		to := Entity{ID: "456", Kind: "user"}
		data := map[string]interface{}{}
		bulkKind := "user"
		bytes, _ := json.Marshal(data)

		Convey("It should have a nil From when the from Entity is empty", func() {
			from = Entity{}
			assoc, err = newAssociation(from, kind, to, data, bulkKind)

			So(assoc.From, ShouldBeNil)
		})

		Convey("It should create an Association with correct parameters", func() {
			assoc, err = newAssociation(from, kind, to, data, bulkKind)

			So(assoc.From, ShouldResemble, &rpc.Entity{Kind: "user", Name: "123"})
		})

		So(assoc.Kind, ShouldEqual, kind)
		So(assoc.To, ShouldResemble, &rpc.Entity{Kind: "user", Name: "456"})
		So(assoc.Data, ShouldResemble, bytes)
		So(assoc.BulkKind, ShouldEqual, bulkKind)
		So(err, ShouldBeNil)
	})
}

func TestNewEntity(t *testing.T) {
	Convey("When creating a new RPC Entity", t, func() {
		Convey("It should have matching parameters in the new rpc Entity", func() {
			e := Entity{ID: "123", Kind: "user"}
			result := newEntity(e)

			So(*result, ShouldResemble, rpc.Entity{Kind: "user", Name: "123"})
		})

		Convey("It should return nil when given an empty Entity", func() {
			e := Entity{}
			result := newEntity(e)

			So(result, ShouldBeNil)
		})
	})
}
